<?php

namespace App\Http\Controllers\Admin;

use App\Models\Category;
use App\Models\City;
use App\Models\ComboProduct;
use App\Models\Media;
use App\Models\Product;
use App\Models\Role;
use App\Models\Seller;
use App\Models\SellerCommission;
use App\Models\SellerStore;
use App\Models\StorageType;
use App\Models\Store;
use App\Models\Transaction;
use App\Models\User;
use App\Models\UserFcm;
use App\Models\Zipcode;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Validator;
use App\Services\TranslationService;
use App\Traits\HandlesValidation;
use App\Services\FirebaseNotificationService;
use App\Services\StoreService;
use App\Services\MediaService;
use App\Services\CurrencyService;

class SellerController extends Controller
{
    use HandlesValidation;
    public function index()
    {
        $users = User::whereHas('role', function ($query) {
            $query->where('id', 2);
        })->get();
        return view('admin.pages.tables.manage_sellers', ['users' => $users]);
    }

    public function update_status($id)
    {
        $user = User::findOrFail($id);
        $user->active = $user->active == '1' ? '0' : '1';
        $user->save();
        return response()->json(['success' => labels('admin_labels.status_updated_successfully', 'Status updated successfully.')]);
    }

    public function create()
    {
        $store_id = app(StoreService::class)->getStoreId();
        $categories = $this->getCategories();

        $categories = Category::where('status', 1)->where('store_id', $store_id)->orderBy('id', 'desc')->get();

        $note_for_necessary_documents = fetchDetails(Store::class, ['id' => $store_id], 'note_for_necessary_documents');
        $note_for_necessary_documents = !$note_for_necessary_documents->isEmpty() ? $note_for_necessary_documents[0]->note_for_necessary_documents : "Other Documents";
        // dD($note_for_necessary_documents);

        $stores = Store::where('status', 1)->get();
        return view('admin.pages.forms.add_sellers', compact('categories', 'stores', 'note_for_necessary_documents'));
    }

    public function store(Request $request, $fromApp = false)
    {
        $rules = [
            'name' => 'required',
            'mobile' => 'required',
            'email' => 'required',
            'password' => 'required',
            'confirm_password' => 'required|same:password',
            'address' => 'required',
            'store_name' => 'required',
            'account_number' => 'required',
            'account_name' => 'required',
            'bank_name' => 'required',
            'bank_code' => 'required',
            'city' => 'required',
            'zipcode' => 'required',
            'description' => 'required',
            'deliverable_type' => 'required',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180'
        ];

        if ($response = $this->HandlesValidation($request, $rules)) {
            return $response;
        }

        $storeImgPath = public_path(config('constants.SELLER_IMG_PATH'));

        if (!File::exists($storeImgPath)) {
            File::makeDirectory($storeImgPath, 0755, true);
        }

        $seller_data = [];
        $seller_store_data = [];
        $store_id = isset($request->store_id) && !empty($request->store_id) ? $request->store_id : app(StoreService::class)->getStoreId();
        $user = User::where('mobile', $request->mobile)->where('role_id', 4)->first();

        $media_storage_settings = fetchDetails(StorageType::class, ['is_default' => 1], '*');
        $mediaStorageType = !$media_storage_settings->isEmpty() ? $media_storage_settings[0]->id : 1;
        $disk = !$media_storage_settings->isEmpty() ? $media_storage_settings[0]->name : 'public';

        $media = StorageType::find($mediaStorageType);
        try {
            if ($request->hasFile('other_documents')) {
                foreach ($request->file('other_documents') as $file) {
                    $other_documents = $media->addMedia($file)
                        ->sanitizingFileName(function ($fileName) use ($media) {
                            $sanitizedFileName = strtolower(str_replace(['#', '/', '\\', ' '], '-', $fileName));
                            $uniqueId = time() . '_' . mt_rand(1000, 9999);
                            $extension = pathinfo($sanitizedFileName, PATHINFO_EXTENSION);
                            $baseName = pathinfo($sanitizedFileName, PATHINFO_FILENAME);

                            return "{$baseName}-{$uniqueId}.{$extension}";
                        })
                        ->toMediaCollection('sellers', $disk);
                    $other_document_file_names[] = $other_documents->file_name;
                    $mediaIds[] = $other_documents->id;
                }
            }
            if ($request->hasFile('profile_image')) {

                $profile_image = $request->file('profile_image');
                $profile_image = $media->addMedia($profile_image)
                    ->sanitizingFileName(function ($fileName) use ($media) {
                        // Replace special characters and spaces with hyphens
                        $sanitizedFileName = strtolower(str_replace(['#', '/', '\\', ' '], '-', $fileName));
                        // Generate a unique identifier based on timestamp and random component
                        $uniqueId = time() . '_' . mt_rand(1000, 9999);
                        $extension = pathinfo($sanitizedFileName, PATHINFO_EXTENSION);
                        $baseName = pathinfo($sanitizedFileName, PATHINFO_FILENAME);

                        return "{$baseName}-{$uniqueId}.{$extension}";
                    })
                    ->toMediaCollection('sellers', $disk);
                $mediaIds[] = $profile_image->id;
            }
            if ($request->hasFile('address_proof')) {

                $addressProofFile = $request->file('address_proof');

                $address_proof = $media->addMedia($addressProofFile)
                    ->sanitizingFileName(function ($fileName) use ($media) {
                        // Replace special characters and spaces with hyphens
                        $sanitizedFileName = strtolower(str_replace(['#', '/', '\\', ' '], '-', $fileName));
                        // Generate a unique identifier based on timestamp and random component
                        $uniqueId = time() . '_' . mt_rand(1000, 9999);
                        $extension = pathinfo($sanitizedFileName, PATHINFO_EXTENSION);
                        $baseName = pathinfo($sanitizedFileName, PATHINFO_FILENAME);

                        return "{$baseName}-{$uniqueId}.{$extension}";
                    })
                    ->toMediaCollection('sellers', $disk);

                $mediaIds[] = $address_proof->id;
            }
            if ($request->hasFile('store_logo')) {

                $storeLogoFile = $request->file('store_logo');

                $store_logo = $media->addMedia($storeLogoFile)
                    ->sanitizingFileName(function ($fileName) use ($media) {
                        // Replace special characters and spaces with hyphens
                        $sanitizedFileName = strtolower(str_replace(['#', '/', '\\', ' '], '-', $fileName));
                        // Generate a unique identifier based on timestamp and random component
                        $uniqueId = time() . '_' . mt_rand(1000, 9999);
                        $extension = pathinfo($sanitizedFileName, PATHINFO_EXTENSION);
                        $baseName = pathinfo($sanitizedFileName, PATHINFO_FILENAME);

                        return "{$baseName}-{$uniqueId}.{$extension}";
                    })
                    ->toMediaCollection('sellers', $disk);

                $mediaIds[] = $store_logo->id;
            }

            if ($request->hasFile('store_thumbnail')) {

                $storeThumbnailFile = $request->file('store_thumbnail');

                $store_thumbnail = $media->addMedia($storeThumbnailFile)
                    ->sanitizingFileName(function ($fileName) use ($media) {
                        // Replace special characters and spaces with hyphens
                        $sanitizedFileName = strtolower(str_replace(['#', '/', '\\', ' '], '-', $fileName));
                        // Generate a unique identifier based on timestamp and random component
                        $uniqueId = time() . '_' . mt_rand(1000, 9999);
                        $extension = pathinfo($sanitizedFileName, PATHINFO_EXTENSION);
                        $baseName = pathinfo($sanitizedFileName, PATHINFO_FILENAME);

                        return "{$baseName}-{$uniqueId}.{$extension}";
                    })
                    ->toMediaCollection('sellers', $disk);

                $mediaIds[] = $store_thumbnail->id;
            }


            if ($request->hasFile('authorized_signature')) {

                $authorizedSignatureFile = $request->file('authorized_signature');

                $authorized_signature = $media->addMedia($authorizedSignatureFile)
                    ->sanitizingFileName(function ($fileName) use ($media) {
                        // Replace special characters and spaces with hyphens
                        $sanitizedFileName = strtolower(str_replace(['#', '/', '\\', ' '], '-', $fileName));
                        // Generate a unique identifier based on timestamp and random component
                        $uniqueId = time() . '_' . mt_rand(1000, 9999);
                        $extension = pathinfo($sanitizedFileName, PATHINFO_EXTENSION);
                        $baseName = pathinfo($sanitizedFileName, PATHINFO_FILENAME);

                        return "{$baseName}-{$uniqueId}.{$extension}";
                    })
                    ->toMediaCollection('sellers', $disk);

                $mediaIds[] = $authorized_signature->id;
            }

            if ($request->hasFile('national_identity_card')) {

                $nationalIdentityCardFile = $request->file('national_identity_card');

                $national_identity_card = $media->addMedia($nationalIdentityCardFile)
                    ->sanitizingFileName(function ($fileName) use ($media) {
                        // Replace special characters and spaces with hyphens
                        $sanitizedFileName = strtolower(str_replace(['#', '/', '\\', ' '], '-', $fileName));
                        // Generate a unique identifier based on timestamp and random component
                        $uniqueId = time() . '_' . mt_rand(1000, 9999);
                        $extension = pathinfo($sanitizedFileName, PATHINFO_EXTENSION);
                        $baseName = pathinfo($sanitizedFileName, PATHINFO_FILENAME);

                        return "{$baseName}-{$uniqueId}.{$extension}";
                    })
                    ->toMediaCollection('sellers', $disk);

                $mediaIds[] = $national_identity_card->id;
            }

            //code for storing s3 object url for media

            if ($disk == 's3') {
                $media_list = $media->getMedia('sellers');
                for ($i = 0; $i < count($mediaIds); $i++) {
                    $media_url = $media_list[($media_list->count()) - (count($mediaIds) - $i)]->getUrl();

                    switch ($i) {
                        case 0:
                            $address_proof_url = $media_url;
                            break;
                        case 1:
                            $logo_url = $media_url;
                            break;
                        case 2:
                            $store_thumbnail_url = $media_url;
                            break;
                        case 3:
                            $authorized_signature_url = $media_url;
                            break;
                        case 4:
                            $national_identity_card_url = $media_url;
                            break;
                        case 5:
                            $profile_image_url = $media_url;
                            break;
                        case 6:
                            $other_documents_url = $media_url;
                            break;
                            // Add more cases as needed
                    }
                    Media::destroy($mediaIds[$i]);
                }
            }
        } catch (Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage(),
            ]);
        }
        $user_data = [
            'role_id' => 4,
            'active' => $request->status,
            'password' => bcrypt($request->password),
            'address' => $request->address,
            'username' => $request->name,
            'mobile' => $request->mobile,
            'email' => $request->email,
            'image' => $disk == 's3' ? (isset($profile_image_url) ? $profile_image_url : '') : (isset($profile_image->file_name) ? '/' . $profile_image->file_name : ''),
        ];
        // dd(json_encode($other_document_file_names));
        $seller_store_data['address_proof'] = $disk == 's3' ? (isset($address_proof_url) ? $address_proof_url : '') : (isset($address_proof->file_name) ? '/' . $address_proof->file_name : '');

        $seller_store_data['logo'] = $disk == 's3' ? (isset($logo_url) ? $logo_url : '') : (isset($store_logo->file_name) ? '/' . $store_logo->file_name : '');

        $seller_store_data['other_documents'] = $disk == 's3' ? (isset($other_documents_url) ? ($other_documents_url) : '') : (isset($other_documents->file_name) ? json_encode($other_document_file_names) : '');

        $seller_store_data['store_thumbnail'] = $disk == 's3' ? (isset($store_thumbnail_url) ? $store_thumbnail_url : '') : (isset($store_thumbnail->file_name) ? '/' . $store_thumbnail->file_name : '');

        $seller_data['authorized_signature'] = $disk == 's3' ? (isset($authorized_signature_url) ? $authorized_signature_url : '') : (isset($authorized_signature->file_name) ? '/' . $authorized_signature->file_name : '');

        $seller_data['national_identity_card'] = $disk == 's3' ? (isset($national_identity_card_url) ? $national_identity_card_url : '') : (isset($national_identity_card->file_name) ? '/' . $national_identity_card->file_name : '');
        // dd($seller_store_data);
        $permmissions = array();
        $permmissions['require_products_approval'] = ($request->require_products_approval == "on") ? 1 : 0;
        $permmissions['customer_privacy'] = ($request->customer_privacy == "on") ? 1 : 0;
        $permmissions['view_order_otp'] = ($request->view_order_otp == "on") ? 1 : 0;
        // dd($request);
        if ($fromApp == true) {
            $requested_categories = $request->requested_categories;
        } else {
            $requested_categories = implode(',', (array) $request->requested_categories);
        }
        if (isset($request->commission_data) && !empty($request->commission_data)) {
            $commission_data = json_decode($request->commission_data, true);
            $category_ids = implode(',', (array) $commission_data['category_id']);
        }



        if (!$user) {
            $user = User::create($user_data);
            if (!empty($request->fcm_id)) {
                $fcm_data = [
                    'fcm_id' => $request->fcm_id ?? '',
                    'user_id' => $user->id,
                ];
                $existing_fcm = UserFcm::where('user_id', $user->id)
                    ->where('fcm_id', $request->fcm_id)
                    ->first();

                if (!$existing_fcm) {
                    UserFcm::insert($fcm_data);
                }
            }

            $seller_data = array_merge($seller_data, [
                'user_id' => $user->id,
                'status' => $request->status ?? 2,
                'pan_number' => $request->pan_number,
                'disk' => isset($authorized_signature->disk) && !empty($authorized_signature->disk) ? $authorized_signature->disk : 'public',
            ]);


            $seller = Seller::create($seller_data);
        } else {
            $seller_store_details = optional($user->stores()->first())->id;
            // dd($seller_store_details);
            $seller = Seller::where('user_id', $user->id)->first();

            if ($seller_store_details == $store_id) {
                return response()->json([
                    'error_message' => labels('admin_labels.seller_already_registered', 'Seller already registered in this store.'),
                    'language_message_key' => 'seller_already_registered'
                ]);
            }
        }
        if ($fromApp == true) {
            $zones = $request->deliverable_zones;
        } else {
            $zones = implode(',', (array) $request->deliverable_zones);
        }
        // dd($fromApp);
        if (isset($request->requested_categories) && !empty($request->requested_categories)) {
            if ($fromApp == true) {
                // $requested_commission_category_ids = implode(',', (array) $request->requested_categories);
                $requested_commission_category_ids = explode(',', $request->requested_categories);
            } else {
                $requested_commission_category_ids =  $request->requested_categories;
            }
            // dd($requested_commission_category_ids);
            foreach ($requested_commission_category_ids as $category_id) {
                SellerCommission::create([
                    'seller_id' => $seller->id,
                    'store_id' => $store_id,
                    'category_id' => $category_id,
                    'commission' => 0,
                ]);
            }
        }
        $seller_store_data = array_merge($seller_store_data, [
            'user_id' => $user->id,
            'seller_id' => $seller->id,
            'store_name' => $request->store_name ?? "",
            'store_url' => $request->store_url ?? "",
            'store_description' => $request->description ?? "",
            'commission' => $request->global_commission ?? 0,
            'account_number' => $request->account_number ?? "",
            'account_name' => $request->account_name ?? "",
            'bank_name' => $request->bank_name ?? "",
            'bank_code' => $request->bank_code ?? "",
            'status' => $request->store_status ? $request->store_status : 0,
            'tax_name' => $request->tax_name ?? "",
            'tax_number' => $request->tax_number ?? "",
            'category_ids' => isset($category_ids) ? $category_ids : ($requested_categories ?? ""),
            'permissions' => (isset($permmissions) && $permmissions != "") ? json_encode($permmissions) : null,
            'slug' => generateSlug($request->input('store_name'), 'seller_store'),
            'store_id' => $store_id,
            'latitude' => $request->latitude ?? "",
            'longitude' => $request->longitude ?? "",
            'city' => $request->city ?? "",
            'zipcode' => $request->zipcode ?? "",
            'disk' => isset($address_proof->disk) && !empty($address_proof->disk) ? $address_proof->disk : 'public',
            'deliverable_type' => isset($request->deliverable_type) && !empty($request->deliverable_type) ? $request->deliverable_type : '',
            'deliverable_zones' => ($request->deliverable_type == '1' || $request->deliverable_type == '0') ? '' : $zones,
        ]);

        $seller_store = SellerStore::insert($seller_store_data);



        if (isset($request->commission_data) && !empty($request->commission_data)) {
            $commission_data = json_decode($request->commission_data, true);
            if (is_array($commission_data['category_id'])) {
                if (count($commission_data['category_id']) >= 2) {
                    $cat_array = array_unique($commission_data['category_id']);
                    foreach ($commission_data['commission'] as $key => $val) {
                        if (!array_key_exists($key, $cat_array))
                            unset($commission_data['commission'][$key]);
                    }
                    $cat_array = array_values($cat_array);
                    $com_array = array_values($commission_data['commission']);

                    for ($i = 0; $i < count($cat_array); $i++) {
                        $tmp['seller_id'] = $seller->id;
                        $tmp['category_id'] = $cat_array[$i];
                        $tmp['commission'] = $com_array[$i];
                        $com_data[] = $tmp;
                    }
                } else {
                    $com_data[0] = array(
                        "seller_id" => $seller->id,
                        "category_id" => $commission_data['category_id'],
                        "commission" => $commission_data['commission'],
                    );
                }
            } else {
                $com_data[0] = array(
                    "seller_id" => $seller->id,
                    "category_id" => $commission_data['category_id'],
                    "commission" => $commission_data['commission'],
                );
            }
        }

        if (isset($com_data) && !empty($com_data)) {
            foreach ($com_data as $commission) {
                // dd($commission);
                SellerCommission::create([
                    'seller_id' => $commission['seller_id'],
                    'store_id' => $store_id,
                    'category_id' => $commission['category_id'],
                    'commission' => $commission['commission'],
                ]);
            }
        }
        if ($request->ajax()) {
            return response()->json([
                'message' => labels('admin_labels.seller_registered_successfully', 'Seller registered successfully'),
                'location' => route('sellers.index')
            ]);
        } else {
            return response()->json($seller);
        }
    }

    public function edit($id)
    {
        $seller_data = User::find($id);
        $store_id = app(StoreService::class)->getStoreId();
        $all_categories = Category::where('status', 1)->where('store_id', $store_id)->orderBy('id', 'desc')->get();

        $zipcodes = Zipcode::orderBy('id', 'desc')->get();

        $cities = City::orderBy('id', 'desc')->get();
        $language_code = app(TranslationService::class)->getLanguageCode();
        $note_for_necessary_documents = fetchDetails(Store::class, ['id' => $store_id], 'note_for_necessary_documents');
        $note_for_necessary_documents = isset($note_for_necessary_documents) && $note_for_necessary_documents[0]->note_for_necessary_documents != null ? $note_for_necessary_documents[0]->note_for_necessary_documents : "Other Documents";

        $store_data = SellerStore::with(['seller', 'zipcode', 'city'])
            ->where('store_id', $store_id)
            ->where('user_id', $id)
            ->get();

        $selected_zipcode_id = $store_data[0]->zipcode ?? null;
        $selected_zipcode_text = (isset($selected_zipcode_id) && !empty($selected_zipcode_id))
            ? fetchDetails(Zipcode::class, ['id' => $selected_zipcode_id], 'zipcode')
            : null;

        $selected_zipcode_text = $selected_zipcode_text[0]->zipcode;
        // dd($store_data[0]->category_ids);
        // dd($store_data[0]->seller->authorized_signature);
        if ($store_data->isEmpty()) {
            return view('admin.pages.views.no_data_found');
        } else {
            $category_ids_string = $store_data[0]->category_ids;
            $existing_category_ids = explode(',', $category_ids_string);
            // dd($existing_category_ids);
            $categories = [];
            foreach ($all_categories as $category) {
                if (!in_array($category->id, $existing_category_ids)) {
                    $categories[] = $category;
                }
            }



            return view('admin.pages.forms.update_seller', compact('seller_data', 'categories', 'store_data', 'store_id', 'zipcodes', 'cities', 'note_for_necessary_documents', 'existing_category_ids', 'language_code', 'selected_zipcode_id', 'selected_zipcode_text'));
        }
    }

    public function update(Request $request, $id, $fromApp = false)
    {
        // dd($request->store_status);

        $seller_data = User::find($id);
        $user = User::find($id);
        $seller_id = Seller::where('user_id', $id)->value('id');

        if (!$seller_data) {
            return response()->json(['error' => true, 'message' => labels('admin_labels.data_not_found', 'Data Not Found')], 404);
        } else {

            $validator = Validator::make($request->all(), [
                'name' => 'required',
                'mobile' => 'required',
                'email' => 'required',
                'address' => 'required',
                'store_name' => 'required',
                'account_number' => 'required',
                'account_name' => 'required',
                'bank_name' => 'required',
                'bank_code' => 'required',
                'city' => 'required',
                'zipcode' => 'required',
                'deliverable_type' => 'required',
                'latitude' => 'sometimes|nullable|numeric|between:-90,90',
                'longitude' => 'sometimes|nullable|numeric|between:-180,180',

            ]);
            if ($fromApp == false) {
                $validator = Validator::make($request->all(), [
                    'global_commission' => 'required',
                ]);
            }
            if ($fromApp == false) {
                $validator = Validator::make($request->all(), [
                    'status' => 'required',
                ]);
            }

            if (!empty($request->input('old')) || !empty($request->input('new'))) {
                $validator = Validator::make($request->all(), [
                    'old' => 'required',
                ]);
            }

            if (!empty($request->input('old'))) {
                if (!Hash::check(($request->input('old')), $user->password)) {
                    if ($request->ajax()) {
                        return response()->json(['message' => labels('admin_labels.incorrect_old_password', 'The old password is incorrect.')], 422);
                    }
                }
            }
            if ($request->filled('new')) {
                $request['password'] = bcrypt($request->input('new'));
            }

            if ($validator->fails()) {
                $errors = $validator->errors();

                if ($request->ajax()) {

                    return response()->json(['errors' => $errors->all()], 422);
                } else {

                    $response = [
                        'error' => true,
                        'message' => $validator->errors(),
                        'code' => 102,
                    ];
                    return response()->json($response);
                }
            }

            if ($fromApp == true) {
                $store_id = $request->store_id;
            } else {
                $store_id = app(StoreService::class)->getStoreId();
            }

            $seller = Seller::find($seller_id);
            $seller_details = Seller::where('user_id', $id)->get();
            $seller_store_detail = SellerStore::where('seller_id', $seller_id)
                ->where('store_id', $store_id)->get();
            $disk = $seller_details[0]->disk;
            $media_storage_settings = fetchDetails(StorageType::class, ['is_default' => 1], '*');
            $current_disk = !$media_storage_settings->isEmpty() ? $media_storage_settings[0]->name : 'public';

            if ($request->hasFile('profile_image')) {

                // Specify the path and disk from which you want to delete the file
                if ($disk == 's3') {
                    $path = $request->edit_profile_image;
                } else {
                    $path = 'sellers/' . $user['image']; // Example path to the file you want to delete
                }

                // Call the removeFile method to delete the file
                app(MediaService::class)->removeMediaFile($path, $disk);

                $profile_image = $request->file('profile_image');

                // Add and sanitize the new image
                $profile_image = $seller->addMedia($profile_image)
                    ->sanitizingFileName(function ($fileName) use ($seller) {
                        // Replace special characters and spaces with hyphens
                        $sanitizedFileName = strtolower(str_replace(['#', '/', '\\', ' '], '-', $fileName));
                        // Generate a unique identifier based on timestamp and random component
                        $uniqueId = time() . '_' . mt_rand(1000, 9999);
                        $extension = pathinfo($sanitizedFileName, PATHINFO_EXTENSION);
                        $baseName = pathinfo($sanitizedFileName, PATHINFO_FILENAME);

                        return "{$baseName}-{$uniqueId}.{$extension}";
                    })
                    ->toMediaCollection('sellers', $current_disk);

                $mediaIds[] = $profile_image->id;

                // If a new image is uploaded, set the image URL for storage
                $imagePath = $disk == 's3'
                    ? (isset($profile_image_url) ? $profile_image_url : '')
                    : (isset($profile_image->file_name) ? '/' . $profile_image->file_name : '');
            } else {
                // If no image is uploaded, keep the existing image path
                $imagePath = $user['image'];
            }
            $user_details = fetchDetails(User::class, ['id' => $id], '*');
            $user_data = [
                'role_id' => 4,
                'active' => $request->status ?? 1,
                'address' => $request->address ?? $user_details[0]->address,
                'username' => $request->name ?? $user_details[0]->username,
                'mobile' => $request->mobile ?? $user_details[0]->mobile,
                'email' => $request->email ?? $user_details[0]->email,
                'image' => $imagePath,
                'city' => $request->city ?? $user_details[0]->city,
                'pincode' => $request->zipcode ?? $user_details[0]->pincode,
            ];
            if ($request->filled('new')) {
                $user_data['password'] = $request->input('password');
            }

            $seller_data->update($user_data);

            // Example disk (filesystem) from which you want to delete the file

            $seller_data = [];
            $seller_store_data = [];

            $mediaIds = [];

            $media_storage_settings = fetchDetails(StorageType::class, ['is_default' => 1], '*');

            try {
                // dd($request->file());
                if ($request->hasFile('other_documents')) {
                    // Retrieve existing files from the database
                    $existing_documents = json_decode($seller_store_detail[0]->other_documents, true) ?? [];

                    $other_documents = $request->file('other_documents');
                    $other_document_file_names = [];

                    foreach ($other_documents as $file) {
                        $uploadedFile = $seller->addMedia($file)
                            ->sanitizingFileName(function ($fileName) {
                                $sanitizedFileName = strtolower(str_replace(['#', '/', '\\', ' '], '-', $fileName));
                                $uniqueId = time() . '_' . mt_rand(1000, 9999);
                                $extension = pathinfo($sanitizedFileName, PATHINFO_EXTENSION);
                                $baseName = pathinfo($sanitizedFileName, PATHINFO_FILENAME);

                                return "{$baseName}-{$uniqueId}.{$extension}";
                            })
                            ->toMediaCollection('sellers', $current_disk);

                        $other_document_file_names[] = $uploadedFile->file_name;
                        $mediaIds[] = $uploadedFile->id;
                    }

                    // Merge new files with existing ones
                    $all_other_documents = array_merge($existing_documents, $other_document_file_names);
                } else {
                    // If no new files are uploaded, keep old ones
                    $all_other_documents = json_decode($seller_store_detail[0]->other_documents, true) ?? [];
                }

                // Store updated list in the database
                $seller_store_data['other_documents'] = json_encode($all_other_documents);

                if ($request->hasFile('address_proof')) {

                    // Specify the path and disk from which you want to delete the file
                    if ($disk == 's3') {
                        $path = $request->edit_address_proof;
                    } else {
                        $path = 'sellers/' . $seller_store_detail[0]->address_proof; // Example path to the file you want to delete
                    }

                    // Call the removeFile method to delete the file
                    app(MediaService::class)->removeMediaFile($path, $disk);


                    $addressProofFile = $request->file('address_proof');

                    $address_proof = $seller->addMedia($addressProofFile)
                        ->sanitizingFileName(function ($fileName) use ($seller) {
                            // Replace special characters and spaces with hyphens
                            $sanitizedFileName = strtolower(str_replace(['#', '/', '\\', ' '], '-', $fileName));
                            // Generate a unique identifier based on timestamp and random component
                            $uniqueId = time() . '_' . mt_rand(1000, 9999);
                            $extension = pathinfo($sanitizedFileName, PATHINFO_EXTENSION);
                            $baseName = pathinfo($sanitizedFileName, PATHINFO_FILENAME);

                            return "{$baseName}-{$uniqueId}.{$extension}";
                        })
                        ->toMediaCollection('sellers', $current_disk);

                    $mediaIds[] = $address_proof->id;
                }
                if ($request->hasFile('store_logo')) {

                    if ($disk == 's3') {
                        $path = $request->edit_store_logo;
                    } else {
                        $path = 'sellers/' . $seller_store_detail[0]->logo; // Example path to the file you want to delete
                    }

                    // Call the removeFile method to delete the file
                    app(MediaService::class)->removeMediaFile($path, $disk);


                    $storeLogoFile = $request->file('store_logo');

                    $store_logo = $seller->addMedia($storeLogoFile)
                        ->sanitizingFileName(function ($fileName) use ($seller) {
                            // Replace special characters and spaces with hyphens
                            $sanitizedFileName = strtolower(str_replace(['#', '/', '\\', ' '], '-', $fileName));
                            // Generate a unique identifier based on timestamp and random component
                            $uniqueId = time() . '_' . mt_rand(1000, 9999);
                            $extension = pathinfo($sanitizedFileName, PATHINFO_EXTENSION);
                            $baseName = pathinfo($sanitizedFileName, PATHINFO_FILENAME);

                            return "{$baseName}-{$uniqueId}.{$extension}";
                        })
                        ->toMediaCollection('sellers', $current_disk);

                    $mediaIds[] = $store_logo->id;
                }
                if ($request->hasFile('store_thumbnail')) {

                    // Check if the old thumbnail exists before attempting to remove it
                    if (!empty($seller_store_detail[0]->store_thumbnail)) {
                        if ($disk == 's3') {
                            $path = $request->edit_store_thumbnail;
                        } else {
                            $path = 'sellers/' . $seller_store_detail[0]->store_thumbnail; // Example path to the file you want to delete
                        }

                        // Call the removeFile method to delete the file
                        app(MediaService::class)->removeMediaFile($path, $disk);
                    }

                    // Proceed with uploading the new store thumbnail
                    $storeThumbnailFile = $request->file('store_thumbnail');
                    // dd($storeThumbnailFile);
                    $store_thumbnail = $seller->addMedia($storeThumbnailFile)
                        ->sanitizingFileName(function ($fileName) use ($seller) {
                            // Replace special characters and spaces with hyphens
                            $sanitizedFileName = strtolower(str_replace(['#', '/', '\\', ' '], '-', $fileName));
                            // Generate a unique identifier based on timestamp and random component
                            $uniqueId = time() . '_' . mt_rand(1000, 9999);
                            $extension = pathinfo($sanitizedFileName, PATHINFO_EXTENSION);
                            $baseName = pathinfo($sanitizedFileName, PATHINFO_FILENAME);

                            return "{$baseName}-{$uniqueId}.{$extension}";
                        })
                        ->toMediaCollection('sellers', $current_disk);

                    // Store the media ID for further reference
                    $mediaIds[] = $store_thumbnail->id;
                }



                if ($request->hasFile('authorized_signature')) {

                    if ($disk == 's3') {
                        $path = $request->edit_authorized_signature;
                    } else {
                        $path = 'sellers/' . $seller->authorized_signature; // Example path to the file you want to delete
                    }

                    // Call the removeFile method to delete the file
                    app(MediaService::class)->removeMediaFile($path, $disk);

                    $authorizedSignatureFile = $request->file('authorized_signature');

                    $authorized_signature = $seller->addMedia($authorizedSignatureFile)
                        ->sanitizingFileName(function ($fileName) use ($seller) {
                            // Replace special characters and spaces with hyphens
                            $sanitizedFileName = strtolower(str_replace(['#', '/', '\\', ' '], '-', $fileName));
                            // Generate a unique identifier based on timestamp and random component
                            $uniqueId = time() . '_' . mt_rand(1000, 9999);
                            $extension = pathinfo($sanitizedFileName, PATHINFO_EXTENSION);
                            $baseName = pathinfo($sanitizedFileName, PATHINFO_FILENAME);

                            return "{$baseName}-{$uniqueId}.{$extension}";
                        })
                        ->toMediaCollection('sellers', $current_disk);

                    $mediaIds[] = $authorized_signature->id;
                }

                if ($request->hasFile('national_identity_card')) {
                    if ($disk == 's3') {
                        $path = $request->edit_national_identity_card;
                    } else {
                        $path = 'sellers/' . $seller->national_identity_card; // Example path to the file you want to delete
                    }

                    // Call the removeFile method to delete the file
                    app(MediaService::class)->removeMediaFile($path, $disk);

                    $nationalIdentityCardFile = $request->file('national_identity_card');

                    $national_identity_card = $seller->addMedia($nationalIdentityCardFile)
                        ->sanitizingFileName(function ($fileName) use ($seller) {
                            // Replace special characters and spaces with hyphens
                            $sanitizedFileName = strtolower(str_replace(['#', '/', '\\', ' '], '-', $fileName));
                            // Generate a unique identifier based on timestamp and random component
                            $uniqueId = time() . '_' . mt_rand(1000, 9999);
                            $extension = pathinfo($sanitizedFileName, PATHINFO_EXTENSION);
                            $baseName = pathinfo($sanitizedFileName, PATHINFO_FILENAME);

                            return "{$baseName}-{$uniqueId}.{$extension}";
                        })
                        ->toMediaCollection('sellers', $current_disk);

                    $mediaIds[] = $national_identity_card->id;
                }


                //code for storing s3 object url for media

                if ($current_disk == 's3') {
                    $media_list = $seller->getMedia('sellers');
                    for ($i = 0; $i < count($mediaIds); $i++) {
                        $media_url = $media_list[($media_list->count()) - (count($mediaIds) - $i)]->getUrl();

                        $fileName = implode('/', array_slice(explode('/', $media_url), -1));

                        if (isset($profile_image->file_name) && $fileName == $profile_image->file_name) {
                            $profile_image_url = $media_url;
                        }
                        if (isset($address_proof->file_name) && $fileName == $address_proof->file_name) {
                            $address_proof_url = $media_url;
                        }
                        if (isset($store_logo->file_name) && $fileName == $store_logo->file_name) {
                            $logo_url = $media_url;
                        }
                        if (isset($store_thumbnail->file_name) && $fileName == $store_thumbnail->file_name) {
                            $store_thumbnail_url = $media_url;
                        }
                        if (isset($authorized_signature->file_name) && $fileName == $authorized_signature->file_name) {
                            $authorized_signature_url = $media_url;
                        }
                        if (isset($national_identity_card->file_name) && $fileName == $national_identity_card->file_name) {
                            $national_identity_card_url = $media_url;
                        }
                        if (isset($other_documents->file_name)) {
                            $other_documents_url = $media_url;
                        }

                        Media::destroy($mediaIds[$i]);
                    }
                }
            } catch (Exception $e) {
                return response()->json([
                    'error' => true,
                    'message' => $e->getMessage(),
                ]);
            }
            if (isset($address_proof->file_name)) {
                $seller_store_data['address_proof'] = $current_disk == 's3' ? (isset($address_proof_url) ? $address_proof_url : '') : (isset($address_proof->file_name) ? '/' . $address_proof->file_name : '');
            } else {
                $seller_store_data['address_proof'] = $request->edit_address_proof;
                $seller_store_data['address_proof'] = $seller_store_detail[0]->address_proof;
            }

            if (isset($store_logo->file_name)) {
                $seller_store_data['logo'] = $current_disk == 's3' ? (isset($logo_url) ? $logo_url : '') : (isset($store_logo->file_name) ? '/' . $store_logo->file_name : '');
            } else {
                $seller_store_data['logo'] = $request->edit_store_logo;
                $seller_store_data['logo'] = $seller_store_detail[0]->logo;
            }

            $seller_store_data['other_documents'] = json_encode($all_other_documents);

            if (isset($store_thumbnail->file_name)) {
                $seller_store_data['store_thumbnail'] = $current_disk == 's3' ? (isset($store_thumbnail_url) ? $store_thumbnail_url : '') : (isset($store_thumbnail->file_name) ? '/' . $store_thumbnail->file_name : '');
            } else {
                $seller_store_data['store_thumbnail'] = $request->edit_store_thumbnail;
                $seller_store_data['store_thumbnail'] = $seller_store_detail[0]->store_thumbnail;
            }
            if (isset($authorized_signature->file_name)) {
                $seller_data['authorized_signature'] = $current_disk == 's3' ? (isset($authorized_signature_url) ? $authorized_signature_url : '') : (isset($authorized_signature->file_name) ? '/' . $authorized_signature->file_name : '');
            } else {
                $seller_data['authorized_signature'] = $request->edit_authorized_signature;
                $seller_data['authorized_signature'] = $seller->authorized_signature;
            }

            if (isset($national_identity_card->file_name)) {
                $seller_data['national_identity_card'] = $current_disk == 's3' ? (isset($national_identity_card_url) ? $national_identity_card_url : '') : (isset($national_identity_card->file_name) ? '/' . $national_identity_card->file_name : '');
            } else {
                $seller_data['national_identity_card'] = $request->edit_national_identity_card;
                $seller_data['national_identity_card'] = $seller->national_identity_card;
            }

            $permmissions = array();
            $permmissions['require_products_approval'] = ($request->require_products_approval == "on") ? 1 : 0;
            $permmissions['customer_privacy'] = ($request->customer_privacy == "on") ? 1 : 0;
            $permmissions['view_order_otp'] = ($request->view_order_otp == "on") ? 1 : 0;

            $commission_data = json_decode($request->commission_data, true);

            if (isset($commission_data['category_id']) && !empty($commission_data['category_id'])) {
                if (isset($commission_data['category_id']) && !empty($commission_data['category_id'])) {
                    if (!is_array($commission_data['category_id'])) {
                        $category_ids = $commission_data['category_id'];
                    } else {
                        $category_ids = implode(',', $commission_data['category_id']);
                    }
                }
            } else {
                $categoryids = fetchDetails(SellerStore::class, ['seller_id' => $seller_id, 'store_id' => $store_id], "*");

                $categories = $categoryids[0]->category_ids;
            }

            $seller_data = array_merge($seller_data, [
                'status' => $request->status ?? $seller->status,
                'pan_number' => $request->pan_number,
                // 'disk' => isset($authorized_signature->disk) && !empty($authorized_signature->disk) ? $authorized_signature->disk : $disk,

            ]);



            $seller->update($seller_data);
            $new_name = $request->store_name;
            $current_name = $seller_store_detail[0]->store_name;
            $current_slug = $seller_store_detail[0]->slug;
            $updated_seller = Seller::where('user_id', $id)->first();

            // send notification to seller when seller's store status change
            if ($fromApp == true) {
                $zones = $request->deliverable_zones;
            } else {
                $zones = implode(',', (array) $request->deliverable_zones);
            }

            // dd($zones);
            $seller_store_data = array_merge($seller_store_data, [
                'store_name' => $request->store_name,
                'store_url' => $request->store_url,
                'store_description' => $request->description,
                'commission' => $request->global_commission ?? 0,
                'account_number' => $request->account_number,
                'account_name' => $request->account_name,
                'bank_name' => $request->bank_name,
                'bank_code' => $request->bank_code,
                'status' => $request->store_status ? $request->store_status : 1,
                'tax_name' => $request->tax_name,
                'tax_number' => $request->tax_number,
                'category_ids' => isset($category_ids) && !empty($category_ids) ? $category_ids : $categories,
                'permissions' => (isset($permmissions) && $permmissions != "") ? json_encode($permmissions) : null,
                'slug' => generateSlug($new_name, 'seller_store', 'slug', $current_slug, $current_name),
                'store_id' => $store_id,
                'latitude' => $request->latitude ?? "",
                'longitude' => $request->longitude ?? "",
                'disk' => isset($address_proof->disk) && !empty($address_proof->disk) ? $address_proof->disk : $disk,
                'city' => $request->city ?? "",
                'zipcode' => $request->zipcode ?? "",
                'deliverable_type' => isset($request->deliverable_type) ? $request->deliverable_type : '',
                'deliverable_zones' => ($request->deliverable_type == '1' || $request->deliverable_type == '0') ? '' : $zones,
            ]);
            if ($request->store_status !== null) {
                if ($seller_store_detail[0] != $request->store_status) {
                    $fcm_ids = array();

                    $results = UserFcm::with('user:id,id,is_notification_on')
                        ->where('user_id', $user->id)
                        ->whereHas('user', function ($q) {
                            $q->where('is_notification_on', 1);
                        })
                        ->get()
                        ->map(function ($fcm) {
                            return [
                                'fcm_id' => $fcm->fcm_id,
                                'is_notification_on' => $fcm->user?->is_notification_on,
                            ];
                        });

                    foreach ($results as $result) {
                        $fcm_ids[] = $result['fcm_id'];
                    }
                    $store_name = fetchDetails(Store::class, ['id' => $store_id], 'name');
                    $store_name = $store_name[0]->name;
                    $store_name = json_decode($store_name, true);
                    $store_name = $store_name['en'] ?? '';
                    // dd($store_name);
                    $status = $request->store_status == 1 ? 'Approved' : 'Not approved';

                    $title = "Store status changed";
                    $message = "Hello dear " . $request->name . " Your " . $store_name . " store status is changed to " . $status;
                    $fcmMsg = array(
                        'title' => "$title",
                        'body' => "$message",
                        'type' => "status_change",
                        'seller_id' => "$seller_id",
                        'store_id' => "$store_id",
                        'status' => "$request->store_status"
                    );
                    $registrationIDs_chunks = array_chunk($fcm_ids, 1000);
                    app(FirebaseNotificationService::class)->sendNotification('', $registrationIDs_chunks, $fcmMsg);
                }
            }
            SellerStore::where('seller_id', $seller_id)
                ->where('store_id', $store_id)
                ->update($seller_store_data);

            $commission_data = json_decode($request->commission_data, true);
            // dd($commission_data);
            if (isset($commission_data['category_id']) && !empty($commission_data['category_id'])) {

                if (is_array($commission_data['category_id'])) {
                    if (count($commission_data['category_id']) >= 2) {
                        $cat_array = array_unique($commission_data['category_id']);
                        foreach ($commission_data['commission'] as $key => $val) {
                            if (!array_key_exists($key, $cat_array))
                                unset($commission_data['commission'][$key]);
                        }
                        $cat_array = array_values($cat_array);
                        $com_array = array_values($commission_data['commission']);

                        for ($i = 0; $i < count($cat_array); $i++) {
                            $tmp['seller_id'] = $updated_seller->id;
                            $tmp['category_id'] = $cat_array[$i];
                            $tmp['commission'] = $com_array[$i];
                            $com_data[] = $tmp;
                        }
                    } else {
                        $com_data[0] = array(
                            "seller_id" => $updated_seller->id,
                            "category_id" => $commission_data['category_id'],
                            "commission" => $commission_data['commission'],
                        );
                    }
                } else {
                    $com_data[0] = array(
                        "seller_id" => $updated_seller->id,
                        "category_id" => $commission_data['category_id'],
                        "commission" => $commission_data['commission'],
                    );
                }
            }


            if (isset($com_data) && !empty($com_data)) {

                deleteDetails(['seller_id' => $updated_seller->id], SellerCommission::class);
                foreach ($com_data as $commission) {
                    // dd($commission);
                    SellerCommission::create([
                        'seller_id' => $commission['seller_id'],
                        'store_id' => $store_id,
                        'category_id' => $commission['category_id'],
                        'commission' => $commission['commission'],
                    ]);
                }
            }
            if ($request->ajax()) {
                return response()->json([
                    'message' => labels('admin_labels.seller_updated_successfully', 'Seller updated successfully'),
                    'location' => route('sellers.index')
                ]);
            }
        }
    }

    public function destroy($id)
    {
        $user = User::find($id);
        // dd($user->id);
        if ($user) {
            $seller_data = Seller::where('user_id', $user->id)->select('id')->get();
            $seller_id = isset($seller_data) ? $seller_data[0]->id : "";
            $products = Product::where('seller_id', $seller_id)->count();
            if ($products > 0) {
                return response()->json([
                    'error' => labels('admin_labels.cannot_delete_seller_associated_data', 'Cannot delete seller. There are associated seller data records.')
                ]);
            }
            Seller::where('user_id', $user->id)->delete();

            if ($user->delete()) {
                return response()->json([
                    'error' => false,
                    'message' => labels('admin_labels.seller_deleted_successfully', 'Seller deleted successfully!')
                ]);
            }
        }
        return response()->json(['error' => labels('admin_labels.data_not_found', 'Data Not Found')]);
    }

    public function list()
    {
        $store_id = app(StoreService::class)->getStoreId();
        $search = trim(request('search'));
        $sort = 'users.id';
        $order = request('order') ?: 'DESC';
        $offset = $search || (request('pagination_offset')) ? (request('pagination_offset')) : 0;
        $allowModification = config('constants.ALLOW_MODIFICATION') == 1;
        $limit = (request('limit')) ? request('limit') : "10";

        $sellers = User::with('seller_data')
            ->select('users.*', 'seller_store.*', 'seller_data.*', 'seller_data.status as seller_status')
            ->where('role_id', 4)
            ->where(function ($query) use ($search) {
                $query->where('users.username', 'like', '%' . $search . '%')
                    ->orWhere('users.id', 'like', '%' . $search . '%')
                    ->orWhere('users.email', 'like', '%' . $search . '%')
                    ->orWhere('users.mobile', 'like', '%' . $search . '%');
            })
            ->join('seller_data', 'users.id', '=', 'seller_data.user_id')
            ->join('seller_store', 'users.id', '=', 'seller_store.user_id')
            ->where('seller_store.store_id', $store_id);

        if (request()->filled('productStatus')) {
            $sellers->where('seller_data.status', request('productStatus'));
        }

        $total = $sellers->count();
        $sellers = $sellers->orderBy($sort, $order)
            ->offset($offset)
            ->limit($limit)
            ->get()
            ->map(function ($seller) use ($allowModification) {

                $isPublicDisk = $seller->disk == 'public' ? 1 : 0;
                $logo = $isPublicDisk
                    ? asset(config('constants.SELLER_IMG_PATH') . $seller->logo)
                    : $seller->logo;

                $store_thumbnail = $isPublicDisk
                    ? asset(config('constants.SELLER_IMG_PATH') . $seller->store_thumbnail)
                    : $seller->store_thumbnail;

                $active_status = "";
                $delete_url = route('admin.sellers.destroy', $seller->user_id);
                $edit_url = route('admin.sellers.edit', $seller->user_id);
                $action = '<div class="dropdown bootstrap-table-dropdown">
                    <a href="#" class="text-dark" data-bs-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                        <i class="bx bx-dots-horizontal-rounded"></i>
                    </a>
                    <div class="dropdown-menu table_dropdown seller_action_dropdown" aria-labelledby="dropdownMenuButton">
                    <a class="dropdown-item dropdown_menu_items" href="' . $edit_url . '"><i class="bx bx-pencil mx-2"></i> Edit</a>
                        <a class="dropdown-item delete-data dropdown_menu_items" data-url="' . $delete_url . '"><i class="bx bx-trash mx-2"></i> Delete</a>
                    </div>
                </div>';
                if ($seller->seller_status == '1') {
                    $active_status = '<label class="badge bg-primary">Approved</label>';
                }
                if ($seller->seller_status == '2') {
                    $active_status = '<label class="badge bg-secondary">Not Approved</label>';
                }
                if ($seller->seller_status == '0') {
                    $active_status = '<label class="badge bg-warning">Deactive</label>';
                }
                if ($seller->seller_status == '7') {
                    $active_status = '<label class="badge bg-secondary">Removed</label>';
                }



                $logo = route('admin.dynamic_image', [
                    'url' => app(MediaService::class)->getMediaImageUrl($seller->logo, 'SELLER_IMG_PATH'),
                    'width' => 60,
                    'quality' => 90
                ]);
                $store_thumbnail = route('admin.dynamic_image', [
                    'url' => app(MediaService::class)->getMediaImageUrl($seller->store_thumbnail, 'SELLER_IMG_PATH'),
                    'width' => 60,
                    'quality' => 90
                ]);

                return [
                    'id' => $seller->id,
                    'name' => $seller->username,
                    'mobile' => $allowModification ? $seller->mobile : '************',
                    'email' => $allowModification ? $seller->email : '************',
                    'balance' => app(CurrencyService::class)->formateCurrency(formatePriceDecimal($seller->balance)),
                    'store_name' => $seller->store_name ?? '',
                    'address' => $seller->address ?? '',
                    'store_url' => $seller->store_url,
                    'store_description' => $seller->store_description,
                    'account_name' => $seller->account_name,
                    'account_number' => $seller->account_number,
                    'bank_name' => $seller->bank_name,
                    'bank_code' => $seller->bank_code,
                    'tax_name' => $seller->tax_name,
                    'tax_number' => $seller->tax_number,
                    'pan_number' => $seller->pan_number,
                    'logo' => '<div class="mx-auto"><a href="' . app(MediaService::class)->getMediaImageUrl($seller->logo, 'SELLER_IMG_PATH') . '" data-lightbox="image-' . $seller->id . '" data-gallery="gallery"><img src="' . $logo . '" class="rounded"></a></div>',
                    'store_thumbnail' => '<div class="mx-auto"><a href="' . app(MediaService::class)->getMediaImageUrl($seller->store_thumbnail, 'SELLER_IMG_PATH') . '" data-lightbox="image-' . $seller->id . '" data-gallery="gallery"><img src="' . $store_thumbnail . '" class="rounded"></a></div>',
                    'status' => $active_status,
                    'operate' => $action
                ];
            });

        return response()->json([
            "rows" => $sellers,
            "total" => $total,
        ]);
    }


    public function getsellerCommissionData(Request $request)
    {
        // dd($request);
        $result = array();
        if (isset($request->id) && !empty($request->id)) {
            $id = $request->id;
            $result = $this->getSellerCommissionDetails($id);
            //dd($result);
            if (empty($result)) {
                if (empty($result)) {
                    $result = $this->getCategories();
                }
            }
        } else {
            $result = fetchDetails(Category::class, ['status' => 1], 'id,name');
        }
        if ((is_array($result) && empty($result)) || ($result instanceof \Illuminate\Support\Collection && $result->isEmpty())) {
            if ($request->ajax()) {
                return response()->json([
                    'error' => 'true',
                    'message' => labels('admin_labels.no_category_commission_data_found', 'No category & commission data found for seller.')
                ]);
            }
        } else {
            if ($request->ajax()) {
                return response()->json(['error' => 'false', 'data' => $result]);
            }
        }
    }

    public function getSellerCommissionDetails($id)
    {
        $store_id = app(StoreService::class)->getStoreId();
        $language_code = app(TranslationService::class)->getLanguageCode();
        $data = SellerCommission::with('category')
            ->where('seller_id', $id)
            ->whereHas('category', function ($query) use ($store_id) {
                $query->where('store_id', $store_id);
            })
            ->orderBy('category_id', 'ASC')
            ->get()
            ->map(function ($commission) use ($language_code) {
                $category = $commission->category;

                return [
                    'id' => $category->id ?? null,
                    'category_id' => $category->id ?? null,
                    'name' => app(TranslationService::class)->getDynamicTranslation(Category::class, 'name', $category->id ?? 0, $language_code),
                    'slug' => $category->slug ?? null,
                    'store_id' => $category->store_id ?? null,
                    'banner' => $category->banner ?? null,
                    'image' => $category->image ?? null,
                    'clicks' => $category->clicks ?? 0,
                    'row_order' => $category->row_order ?? null,
                    'status' => $category->status ?? null,
                    'parent_id' => $category->parent_id ?? 0,
                    'style' => $category->style ?? null,
                    'created_at' => $category->created_at ?? null,
                    'updated_at' => $category->updated_at ?? null,
                    'seller_id' => $commission->seller_id,
                    'commission' => $commission->commission,
                ];
            })
            ->toArray();


        if (!empty($data)) {
            foreach ($data as &$item) {
                $item['name'] = app(TranslationService::class)->getDynamicTranslation(Category::class, 'name', $item['category_id'], $language_code);
            }
            return $data;
        } else {
            return false;
        }
    }
    public function getCategories(
        $id = null,
        $limit = null,
        $offset = null,
        $sort = 'row_order',
        $order = 'ASC',
        $has_child_or_item = 'true',
        $slug = '',
        $ignore_status = '',
        $seller_id = '',
        $store_id = '',
        $language_code = ""
    ) {
        // dd('here');
        $level = 0;

        $storeId = app(StoreService::class)->getStoreId();

        $query = Category::query();

        // Apply store filters
        if (!empty($storeId)) {
            $query->where('store_id', $storeId);
        }
        if (!empty($store_id)) {
            $query->where('store_id', $store_id);
        }

        // Filter by ID
        if (!empty($id)) {
            $query->where('id', $id);
            if ($ignore_status != 1) {
                $query->where('status', 1);
            }
        } else {
            if ($ignore_status != 1) {
                $query->where('status', 1);
            }
        }

        // Filter by slug
        if (!empty($slug)) {
            $query->where('slug', $slug);
        }

        // If has_child_or_item = false, filter categories with children or products
        if ($has_child_or_item === 'false') {
            // Use whereHas for children or products
            $query->where(function ($q) {
                $q->whereHas('children')
                    ->orWhereHas('products');  // Assuming Category has products() relation
            });
        }

        // Pagination
        if (!is_null($offset)) {
            $query->skip($offset);
        }
        if (!is_null($limit)) {
            $query->take($limit);
        }

        // Sorting
        $query->orderBy($sort, $order);

        // Eager load children
        $categories = $query->with(['children' => function ($q) {
            $q->where('status', 1);
        }])->get();

        $countRes = $categories->count();

        // Map categories to add translations and other metadata
        $categories = $categories->map(function ($category) use ($language_code, $level) {
            $category->children = $this->formatSubCategories($category->children, $language_code, $level + 1);

            $category->text = app(TranslationService::class)->getDynamicTranslation(Category::class, 'name', $category->id, $language_code);
            $category->name = $category->text;
            $category->state = ['opened' => true];
            $category->icon = "jstree-folder";
            $category->level = $level;
            $category->image = app(MediaService::class)->dynamic_image(app(MediaService::class)->getImageUrl($category->image, 'thumb', 'sm'), 400);
            $category->banner = app(MediaService::class)->dynamic_image(app(MediaService::class)->getImageUrl($category->banner, 'thumb', 'md'), 400);

            return $category;
        });

        if ($categories->isNotEmpty()) {
            $categories[0]->total = $countRes;
        }

        return Response::json(compact('categories', 'countRes'));
    }
    private function formatSubCategories($subCategories, $language_code, $level)
    {
        return $subCategories->map(function ($category) use ($language_code, $level) {
            $category->children = $this->formatSubCategories($category->children, $language_code, $level + 1);
            $category->text = app(TranslationService::class)->getDynamicTranslation(Category::class, 'name', $category->id, $language_code);
            $category->name = $category->text;
            $category->state = ['opened' => true];
            $category->icon = "jstree-folder";
            $category->level = $level;
            $category->image = app(MediaService::class)->dynamic_image(app(MediaService::class)->getImageUrl($category->image, 'thumb', 'sm'), 400);
            $category->banner = app(MediaService::class)->dynamic_image(app(MediaService::class)->getImageUrl($category->banner, 'thumb', 'md'), 400);
            return $category;
        });
    }

    public function getSellers($zipcode_id = "", $limit = null, $offset = '', $sort = 'users.id', $order = 'DESC', $search = null, $filter = [], $store_id = '', $seller_ids = '', $user_id = '')
    {
        $query = User::with(['sellerStore', 'favorites' => function ($q) use ($user_id) {
            $q->where('user_id', $user_id);
        }])
            ->where('active', 1)
            ->where('role_id', 4)
            ->whereHas('sellerStore', function ($q) use ($store_id, $filter) {
                $q->where('status', 1)->where('store_id', $store_id);

                if (!empty($filter['slug'])) {
                    $q->where('slug', $filter['slug']);
                }

                if (request()->filled('seller_id')) {
                    $q->where('seller_id', request()->input('seller_id'));
                }
            });

        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('id', 'LIKE', "%{$search}%")
                    ->orWhere('username', 'LIKE', "%{$search}%")
                    ->orWhere('email', 'LIKE', "%{$search}%")
                    ->orWhere('mobile', 'LIKE', "%{$search}%")
                    ->orWhere('address', 'LIKE', "%{$search}%")
                    ->orWhere('balance', 'LIKE', "%{$search}%")
                    ->orWhereHas('sellerStore', function ($sq) use ($search) {
                        $sq->where('store_name', 'LIKE', "%{$search}%");
                    });
            });
        }

        // Filter by zipcode deliverability
        if (!empty($zipcode_id)) {
            $query->whereHas('sellerStore', function ($q) use ($zipcode_id) {
                $q->where(function ($subQ) use ($zipcode_id) {
                    $subQ->where(function ($zipQ) use ($zipcode_id) {
                        $zipQ->where('deliverable_type', 2)
                            ->whereRaw("FIND_IN_SET(?, deliverable_zipcodes)", [$zipcode_id]);
                    })->orWhere('deliverable_type', 1)
                        ->orWhere(function ($zipQ) use ($zipcode_id) {
                            $zipQ->where('deliverable_type', 3)
                                ->whereRaw("NOT FIND_IN_SET(?, deliverable_zipcodes)", [$zipcode_id]);
                        });
                });
            });
        }

        // Filter by specific seller_ids
        if (!empty($seller_ids) && is_array($seller_ids)) {
            $query->whereHas('sellerStore', function ($q) use ($seller_ids) {
                $q->whereIn('seller_id', $seller_ids);
            });
        }

        $total = $query->count();

        $query->orderBy($sort, $order);

        if ($limit !== null) {
            $query->limit($limit);
        }

        if ($offset !== null) {
            $query->offset($offset);
        }

        $results = $query->get();

        $bulkData = [
            'error' => $results->isEmpty(),
            'message' => $results->isEmpty() ? labels('admin_labels.sellers_not_exist', 'Seller(s) does not exist')
                : labels('admin_labels.seller_retrieved_successfully', 'Seller retrieved successfully'),
            'language_message_key' => $results->isEmpty() ? 'sellers_not_exist' : 'seller_retrived_successfully',
            'total' => $total,
            'data' => [],
        ];

        $rows = [];

        foreach ($results as $user) {
            $sellerStore = $user->sellerStore;

            $regularProductsCount = Product::where('seller_id', $sellerStore->seller_id)
                ->where('store_id', $store_id)
                ->where('status', 1)
                ->count();

            $comboProductsCount = ComboProduct::where('seller_id', $sellerStore->seller_id)
                ->where('store_id', $store_id)
                ->where('status', 1)
                ->count();

            $tempRow = [
                'seller_id' => $sellerStore->seller_id,
                'user_id' => $user->id,
                'seller_name' => stripslashes($user->username),
                'email' => $user->email,
                'mobile' => $user->mobile,
                'slug' => $sellerStore->slug ?? '',
                'rating' => $sellerStore->rating ?? '',
                'no_of_ratings' => $sellerStore->no_of_ratings ?? '',
                'store_name' => stripslashes($sellerStore->store_name),
                'store_url' => stripslashes($sellerStore->store_url),
                'store_description' => stripslashes($sellerStore->store_description),
                'store_logo' => app(MediaService::class)->getMediaImageUrl($sellerStore->logo, 'SELLER_IMG_PATH'),
                'balance' => empty($user->balance) ? "0" : number_format($user->balance, 2),
                'total_products' => $regularProductsCount + $comboProductsCount,
                'is_favorite' => $user->favorites->isNotEmpty() ? 1 : 0,
            ];

            $rows[] = $tempRow;
        }

        $bulkData['data'] = $rows;

        return $bulkData;
    }


    public function sellerWallet()
    {
        return view('admin.pages.tables.seller_wallet');
    }

    public function wallet_transactions_list($user_id = '', $role_id = 2)
    {
        $search = trim(request()->input('search')) ?? '';
        $offset = $search || request()->filled('pagination_offset') ? request('pagination_offset') : 0;
        $limit = request()->input('limit', 10);
        $sort = request()->input('sort', 'id');
        $order = request()->input('order', 'ASC');
        $user_id = $user_id ?: request()->input('user_id', '');

        $transactionsQuery = Transaction::with('user');

        // Filter by transaction_type
        if (request()->filled('transaction_type')) {
            $transactionsQuery->where('transaction_type', request()->input('transaction_type'));
        }

        // Filter by user ID
        if (!empty($user_id)) {
            $transactionsQuery->where('user_id', $user_id);
        }

        // Filter by user_type (role name)
        if (request()->filled('user_type')) {
            $roleName = request()->input('user_type');
            $role_id = Role::where('name', $roleName)->value('id');
            $transactionsQuery->whereHas('user', function ($q) use ($role_id) {
                $q->where('role_id', $role_id);
            });
        }

        // Filter by date range
        if (request()->filled('start_date') && request()->filled('end_date')) {
            $transactionsQuery->whereBetween('created_at', [
                request()->input('start_date'),
                request()->input('end_date'),
            ]);
        }

        // Search logic
        if (!empty($search)) {
            $transactionsQuery->where(function ($q) use ($search) {
                $q->where('id', 'LIKE', "%{$search}%")
                    ->orWhere('amount', 'LIKE', "%{$search}%")
                    ->orWhere('created_at', 'LIKE', "%{$search}%")
                    ->orWhere('type', 'LIKE', "%{$search}%")
                    ->orWhere('status', 'LIKE', "%{$search}%")
                    ->orWhere('txn_id', 'LIKE', "%{$search}%")
                    ->orWhereHas('user', function ($uq) use ($search) {
                        $uq->where('username', 'LIKE', "%{$search}%")
                            ->orWhere('email', 'LIKE', "%{$search}%")
                            ->orWhere('mobile', 'LIKE', "%{$search}%");
                    });
            });
        }

        // Total count
        $total = $transactionsQuery->count();

        // Paginate and fetch results
        $transactions = $transactionsQuery->orderBy($sort, $order)
            ->skip($offset)
            ->take($limit)
            ->get();

        $formattedTransactions = $transactions->map(function ($row) {
            $operate = '';
            if ($row->type === 'bank_transfer') {
                $operate = '<div class="d-flex align-items-center">
                    <a class="single_action_button edit_transaction"
                        data-id="' . $row->id . '"
                        data-txn_id="' . $row->txn_id . '"
                        data-status="' . $row->status . '"
                        data-message="' . $row->message . '"
                        data-bs-target="#transaction_modal"
                        data-bs-toggle="modal">
                        <i class="bx bx-pencil mx-2"></i>
                    </a>
                </div>';
            }

            return [
                'id' => $row->id,
                'name' => $row->user->username ?? '',
                'type' => $row->type === 'bank_transfer' ? 'Bank Transfer' : $row->type,
                'order_id' => $row->order_id,
                'txn_id' => $row->txn_id,
                'payu_txn_id' => $row->payu_txn_id,
                'amount' => $row->amount,
                'status' => $row->status,
                'message' => $row->message,
                'created_at' => Carbon::parse($row->created_at)->format('d-m-Y'),
                'operate' => $operate,
            ];
        });

        return response()->json([
            'total' => $total,
            'rows' => $formattedTransactions
        ]);
    }


    public function seller_wallet_transactions_list($user_id = '')
    {
        $search = trim(request()->input('search')) ?? '';
        $offset = request()->filled('pagination_offset') ? request('pagination_offset') : 0;
        $limit = request()->input('limit', 10);
        $sort = request()->input('sort', 'id');
        $order = request()->input('order', 'ASC');
        $user_id = $user_id ?: request()->input('user_id');

        $store_id = app(StoreService::class)->getStoreId();
        if (empty($store_id)) {
            return response()->json(['error' => 'Store ID is not found in the session'], 400);
        }

        $transactionsQuery = Transaction::with(['user', 'orderItem'])
            ->whereHas('user', function ($q) {
                $q->where('role_id', 4); // Seller
            })
            ->whereHas('orderItem', function ($q) use ($store_id) {
                $q->where('store_id', $store_id);
            });

        // Filter by transaction type
        if (request()->filled('transaction_type')) {
            $transactionsQuery->where('transaction_type', request()->input('transaction_type'));
        }

        // Filter by user ID
        if (!empty($user_id)) {
            $transactionsQuery->where('user_id', $user_id);
        }

        // Filter by search
        if (!empty($search)) {
            $transactionsQuery->where(function ($q) use ($search) {
                $q->where('id', 'LIKE', "%$search%")
                    ->orWhere('amount', 'LIKE', "%$search%")
                    ->orWhere('txn_id', 'LIKE', "%$search%")
                    ->orWhere('type', 'LIKE', "%$search%")
                    ->orWhere('status', 'LIKE', "%$search%")
                    ->orWhere('message', 'LIKE', "%$search%")
                    ->orWhereDate('created_at', '=', $search)
                    ->orWhereHas('user', function ($uq) use ($search) {
                        $uq->where('username', 'LIKE', "%$search%")
                            ->orWhere('mobile', 'LIKE', "%$search%")
                            ->orWhere('email', 'LIKE', "%$search%");
                    });
            });
        }

        // Filter by date range
        if (request()->filled('start_date') && request()->filled('end_date')) {
            $transactionsQuery->whereBetween('created_at', [
                request()->input('start_date'),
                request()->input('end_date'),
            ]);
        }

        // Clone for total
        $total = $transactionsQuery->count();

        // Fetch paginated results
        $transactions = $transactionsQuery->orderBy($sort, $order)
            ->skip($offset)
            ->take($limit)
            ->get();

        // Format the results
        $formattedTransactions = $transactions->map(function ($row) {
            $operate = '';
            if ($row->type === 'bank_transfer') {
                $operate = '<div class="dropdown bootstrap-table-dropdown">
                    <a href="#" class="text-dark" data-bs-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                        <i class="bx bx-dots-horizontal-rounded"></i>
                    </a>
                    <div class="dropdown-menu table_dropdown" aria-labelledby="dropdownMenuButton">
                        <a class="dropdown-item dropdown_menu_items edit_transaction"
                            data-id="' . $row->id . '"
                            data-txn_id="' . $row->txn_id . '"
                            data-status="' . $row->status . '"
                            data-message="' . $row->message . '"
                            data-bs-target="#transaction_modal"
                            data-bs-toggle="modal">
                            <i class="bx bx-pencil mx-2"></i>Edit
                        </a>
                    </div>
                </div>';
            }

            return [
                'id' => $row->id,
                'name' => $row->user->username ?? '',
                'type' => $row->type === 'bank_transfer' ? 'Bank Transfer' : $row->type,
                'order_id' => $row->order_id,
                'txn_id' => $row->txn_id,
                'payu_txn_id' => $row->payu_txn_id,
                'amount' => $row->amount,
                'status' => $row->status,
                'message' => $row->message,
                'created_at' => Carbon::parse($row->created_at)->format('d-m-Y'),
                'operate' => $operate,
            ];
        });

        return response()->json([
            'total' => $total,
            'rows' => $formattedTransactions,
        ]);
    }


    public function delete_selected_data(Request $request)
    {
        // Validate the incoming request
        $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:seller_data,id' // Validate seller IDs
        ]);

        // Initialize arrays to track deletable and non-deletable seller IDs
        $nonDeletableIds = [];
        $deletedSellers = [];

        // Loop through each seller ID
        foreach ($request->ids as $sellerId) {
            // Get the seller based on the provided seller ID
            $seller = Seller::find($sellerId);

            if ($seller) {
                // Get the associated user for the seller
                $user = User::find($seller->user_id);

                if ($user) {
                    // Check if there are any associated products with the seller ID
                    $productsCount = Product::where('seller_id', $seller->id)->count();

                    if ($productsCount > 0) {
                        // If there are associated products, collect the seller ID
                        $nonDeletableIds[] = $seller->id;
                    } else {
                        // Delete the seller
                        if ($seller->delete()) {
                            $deletedSellers[] = $seller->id;
                        }
                    }
                }
            }
        }

        // Check if there are any non-deletable sellers
        if (!empty($nonDeletableIds)) {
            return response()->json([
                'error' => labels(
                    'admin_labels.cannot_delete_seller_associated_data',
                    'Cannot delete the following sellers: ' . implode(', ', $nonDeletableIds) . ' because they have associated products.'
                ),
                'non_deletable_ids' => $nonDeletableIds
            ], 401);
        }

        // If all sellers were deleted successfully, return success message
        return response()->json([
            'error' => false,
            'message' => labels('admin_labels.seller_deleted_successfully', 'Selected sellers deleted successfully!'),
            'deleted_ids' => $deletedSellers
        ]);
    }
    public function get_seller_deliverable_type(Request $request)
    {
        $store_id = app(StoreService::class)->getStoreId();
        $seller_id = isset($request->seller_id) ? $request->seller_id : "";
        // dd($seller_id);
        $deliverable_type = fetchDetails(SellerStore::class, ['seller_id' => $seller_id, 'store_id' => $store_id], ['deliverable_type', 'deliverable_zones']);
        $deliverable_type = !$deliverable_type->isEmpty() ? $deliverable_type[0] : [];
        // dd($deliverable_type);
        return response()->json(['deliverable_type' => $deliverable_type->deliverable_type]);
    }
}
