<?php

namespace App\Libraries;

use Exception;
use Illuminate\Support\Facades\Log;
use Juspay\JuspayEnvironment;
use Juspay\Model\JuspayJWT;
use Juspay\Model\Order;
use Juspay\Model\OrderSession;
use Juspay\RequestOptions;
use Juspay\Exception\JuspayException;

class HDFC
{
    protected $config;
    protected $initError = null;

    public function __construct()
    {
        try{
            $this->config = config('hdfc');

            $privateKey = array_key_exists("PRIVATE_KEY", $this->config) ? $this->config["PRIVATE_KEY"] : null;
            $publicKey = array_key_exists("PUBLIC_KEY", $this->config) ? $this->config["PUBLIC_KEY"] : null;

            if (!$privateKey && isset($this->config["PRIVATE_KEY_PATH"]) && file_exists(base_path($this->config["PRIVATE_KEY_PATH"]))) {
                $privateKey = file_get_contents(base_path($this->config["PRIVATE_KEY_PATH"]));
            }

            if (!$publicKey && isset($this->config["PUBLIC_KEY_PATH"]) && file_exists(base_path($this->config["PUBLIC_KEY_PATH"]))) {
                $publicKey = file_get_contents(base_path($this->config["PUBLIC_KEY_PATH"]));
            }

            if (!$privateKey || !$publicKey) {
                throw new Exception("Juspay keys not found. Check PRIVATE_KEY_PATH/PUBLIC_KEY_PATH in config/env.");
            }
            JuspayEnvironment::init()
                ->withBaseUrl($this->config['base_url'])
//                ->withBaseUrl(JuspayEnvironment::SANDBOX_BASE_URL)
                ->withMerchantId($this->config['merchant_id'])
                ->withJuspayJWT(new JuspayJWT(
                    $this->config['key_uuid'],
                    $publicKey,
                    $privateKey
                ));
        } catch (Exception $e){
            Log::error("HDFC Init Error: " . $e->getMessage());
            $this->initError = $e->getMessage();
        }

    }

    public function createSession($orderId, $amount = 1, $customerId = "test-customer", $customerEmail = null, $customerMobile = null)
    {
        if ($this->initError) {
            return [
                'error' => true,
                'message' => "HDFC Initialization Failed: " . $this->initError,
                'code' => 500
            ];
        }

        try {

            $params = [
                'amount' => $amount,
                'order_id' => env('HDFCORDERPRE','').$orderId,
                'merchant_id' => $this->config['merchant_id'],
                'customer_id' => $customerId,
                'currency'=>'INR',
                'payment_page_client_id' => $this->config['PAYMENT_PAGE_CLIENT_ID'],
                'action' => 'paymentPage',
                'email' => $customerEmail ?? 'mailwdfwd@gami.com',
                'return_url' => route('hdfc.callback')
//            route('hdfc.callback'),
            ];
            
            if ($customerMobile) {
                $params['mobile_number'] = $customerMobile;
            }

//            dd($params);
            $requestOption = new RequestOptions();
            $requestOption->withCustomerId($customerId);

            $session = OrderSession::create($params, $requestOption);
            return [
                'error' => false,
                'session_id' => $session->id,
                'order_id' => $session->orderId,
                'status' => $session->status,
                'payment_link' => $session->paymentLinks['web'] ?? null,
                'sdk_payload' => $session->sdkPayload,
            ];
        } catch (JuspayException $e) {
//            dd($e);
            Log::error("Juspay API Error: " . $e->getErrorMessage());
            Log::error($e->getHttpResponseCode()); 
            return [
                'error' => true,
                'trace'=>$e->getTrace(),
                'message' => $e->getErrorMessage(),
                'http_code' => $e->getHttpResponseCode(),
                'code' => $e->getErrorCode(),
            ];
        } catch (Exception $e) {
             Log::error("General Exception in createSession: " . $e->getMessage());
             return [
                'error' => true,
                'message' => $e->getMessage(),
                'code' => 500,
             ];
        }
    }

    public function getOrderStatus($orderId, $customerId = "test-customer")
    {
        if ($this->initError) {
             return [
                'error' => true,
                'message' => "HDFC Initialization Failed: " . $this->initError,
            ];
        }

        try {
            $params = array();
            $params = ['order_id' => $orderId];

            $requestOption = new RequestOptions();
            $requestOption->withCustomerId($customerId);

//            dd($params,$requestOption);
            $order = Order::status($params, $requestOption);
             Log::info('Juspay Order id : ' .$order->orderId );
            Log::info('Juspay amount : ' .$order->amount );

            Log::info('Juspay status : ' .$order->status );
            $response= $order->__get('*');

            Log::info($response);
            $txnId = '';
            if (isset($order->txnId)) {
                // txnId exists and is not null
                Log::info('Juspay txnId : ' .$order->txnId );
                $txnId=$order->txnId;
            }
            return [
                'order_id' => $order->orderId,
                'amount' => $order->amount,
                'txnId' =>$txnId,
                'status' => $order->status,
                'error'=>false
            ];
        } catch (JuspayException $e) {
            Log::error( "error code" . $e->getHttpResponseCode() . PHP_EOL);
            Log::error( "error message: " . $e->getErrorMessage() . PHP_EOL);
            Log::error( "error code" . $e->getErrorCode() . PHP_EOL);
            Log::error('Juspay error : ' .$e->getErrorMessage() );
            return [
                'error' => true,
                'message' => $e->getErrorMessage(),
                'code' => $e->getErrorCode(),

            ];
        }
    }
}
