<?php

namespace App\Livewire\Orders;

use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Pagination\LengthAwarePaginator;
use App\Services\OrderService;

use Livewire\Attributes\Url;

class Listing extends Component
{
    #[Url]
    public $orderStatus = "";
    
    #[Url(as: 'page')]
    public $currentPage = 1;

    public function render()
    {
        $store_id = session('store_id');
        $user = Auth::user();

        if (!$user) {
            abort(404);
        }

        $perPage = 8;
        $user_orders = $this->getOrders($user->id, $store_id, $perPage);

        return view('livewire.' . config('constants.theme') . '.orders.listing', [
            'user_info' => $user,
            'user_orders' => $user_orders,
            'currentPage' => $this->currentPage,
            'perPage' => $perPage,
        ])->title("Orders |");
    }

    public function getOrders($userId, $store_id, $perPage)
    {
        $order_status = null;
        if (!empty($this->orderStatus)) {
            $order_status = explode(',', $this->orderStatus);
            $order_status = array_map('trim', $order_status);
        }
        
        $orders = app(OrderService::class)->fetchOrders(
            user_id: $userId,
            status: $order_status,
            sort: "o.id",
            order: 'DESC',
            store_id: $store_id
        );

        $ordersCollection = collect($orders['order_data'] ?? []);
        $totalResults = $orders['total'] ?? 0;
        $page = $this->currentPage;

        $paginator = new LengthAwarePaginator(
            $ordersCollection->forPage($page, $perPage),
            $totalResults,
            $perPage,
            $page,
            ['path' => url()->current()]
        );

        return [
            'order_data' => $paginator->items(),
            'total' => $totalResults,
        ];
    }

    public function filterOrders($status)
    {
        $this->orderStatus = $status;
        $this->currentPage = 1;
    }

    public function goToPage($page)
    {
        $this->currentPage = $page;
    }
}
