<?php

namespace App\Livewire\Products;

use App\Models\Attribute_values;
use App\Models\Brand;
use App\Models\Category;
use App\Models\Product;
use App\Models\Section;
use App\Services\MediaService;
use Illuminate\Http\Request;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\Auth;
use App\Services\ProductService;
use Livewire\Attributes\Title;
use Livewire\Component;
use App\Services\TranslationService;
use App\Services\SettingService;
class Listing extends Component
{
    #[Title('Product Listing |')]
    public $user_id;

    public $slug;
    public $routeType = "";
    public $section;
    public $category;
    public $relative_products = [];
    public function __construct()
    {
        $this->user_id = Auth::user() != '' ? Auth::user()->id : NUll;
    }

    function mount(Request $request, $slug = null, $section = null)
    {
        $this->slug = $slug;
        $url = $request->url();
        $store_id = session('store_id');
        if (str_contains($url, '/section/')) {
            $this->routeType = 'section';
            if ($this->slug != "") {
                $this->section = fetchDetails(Section::class, ['id' => $this->slug, 'store_id' => $store_id]);
                if ($this->section == []) {
                    $this->redirect('products', true);
                }
            }
        } elseif (str_contains($url, '/categories/')) {
            $this->routeType = 'category';
            if ($this->slug != "") {
                // Use query builder pattern like CategoryProductController
                $category = Category::query()
                    ->when($store_id, fn($q) => $q->where('store_id', $store_id))
                    ->where('slug', $this->slug)
                    ->first();

                if (!$category) {
                    $this->redirect('products', true);
                } else {
                    $this->category = [$category];
                }
            }
        }
    }

    public function render(Request $request)
    {
        // dd($request);
        $system_settings = app(SettingService::class)->getSettings('system_settings', true);
        $system_settings = json_decode($system_settings, true);
        // Currency context for price filtering
        $currency_code = $system_settings['currency_setting']['code'] ?? null;
        $currency_details = $currency_code ? app(\App\Services\CurrencyService::class)->getCurrencyCodeSettings($currency_code) : [];
        $exchange_rate = isset($currency_details[0]['exchange_rate']) ? (float) $currency_details[0]['exchange_rate'] : 1;
        $right_breadcrumb = [];
        $language_code = app(TranslationService::class)->getLanguageCode();
        $filter = [];
        $store_id = session('store_id');
        // category filter
        $category_id = null;
        $sub_categories = [];
        $currentCategorySlug = null;
        
        // category filter
        $category_slugs = $request->query('category');
        $selected_category_ids = [];
        if (!empty($category_slugs)) {
            $slugs_array = explode(',', $category_slugs);
            $selected_category_ids = Category::whereIn('slug', $slugs_array)
                ->where('store_id', $store_id)
                ->pluck('id')
                ->toArray();
        }

        if (empty($selected_category_ids) && $this->category != []) {
             // Fallback to route slug if no multiple selection in query
             $category = $this->category;
             $selected_category_ids = [$category[0]->id];
        }

        if ($this->category != []) {
             $category = $this->category;
             $sub_categories = Category::query()
                ->when($store_id, fn($q) => $q->where('store_id', $store_id))
                ->where('parent_id', $category[0]->id)
                ->where('status', 1)
                ->get()
                ->map(function ($c) {
                    $c->image_url = route('admin.dynamic_image', [
                        'url' => app(MediaService::class)->getMediaImageUrl($c->image),
                        'width' => 365,
                        'quality' => 90,
                    ]);
                    return $c;
                });
        }

        if (!empty($selected_category_ids)) {
            // Fetch all descendants for filtering
            $allStoreCategoriesForFilter = Category::select('id', 'parent_id')
                ->where('store_id', $store_id)
                ->where('status', 1)
                ->get();

            $getDescendants = function ($parentId) use ($allStoreCategoriesForFilter, &$getDescendants) {
                $ids = [];
                foreach ($allStoreCategoriesForFilter as $cat) {
                    if ($cat->parent_id == $parentId) {
                        $ids[] = $cat->id;
                        $ids = array_merge($ids, $getDescendants($cat->id));
                    }
                }
                return $ids;
            };

            $all_ids_to_filter = $selected_category_ids;
            foreach ($selected_category_ids as $sid) {
                $all_ids_to_filter = array_merge($all_ids_to_filter, $getDescendants($sid));
            }
            $category_id = array_unique($all_ids_to_filter);
        }

        // Build Hierarchical Tree for UI
        $allCatsForTree = Category::query()
            ->select('id', 'name', 'slug', 'parent_id', 'row_order')
            ->when($store_id, fn($q) => $q->where('store_id', $store_id))
            ->where('status', 1)
            ->orderBy('row_order')
            ->orderBy('name')
            ->get();

        // Recursive function for tree building
        $buildTree = function ($parentId = 0) use ($allCatsForTree, &$buildTree, $language_code, $category_slugs) {
            $selected_slugs = explode(',', $category_slugs ?? '');
            $branch = [];
            foreach ($allCatsForTree as $cat) {
                if ($cat->parent_id == $parentId) {
                    $cat->name = app(TranslationService::class)->getDynamicTranslation(Category::class, 'name', $cat->id, $language_code);
                    $children = $buildTree($cat->id);
                    $cat->children = $children;
                    $cat->is_checked = in_array($cat->slug, $selected_slugs);
                    
                    $branch[] = $cat;
                }
            }
            return $branch;
        };

        $filterCategories = $buildTree(0);

        // section filter
        $section = $this->section;
        if ($section != []) {
            if ($section[0]->product_type == 'custom_products') {
                $product_ids = explode(",", $section[0]->product_ids);
                $product_variant_ids = [];
                $product_variants = app(ProductService::class)->fetchProduct(null, null, $product_ids);
                if (count($product_variants) >= 1) {
                    foreach ($product_variants['product'] as $product) {
                        array_push($product_variant_ids, $product->variants[0]['id']);
                    }
                }
                $filter['product_variant_ids'] = $product_variant_ids;
            } else {
                // $category_id = explode(",", $section[0]->categories);
                $filter['category_ids'] = array_map('intval', explode(',', $section[0]->categories));

                $filter['product_type'] = $section[0]->product_type;
            }
            $breadcrumb = 'Section <ion-icon class="align-text-top icon"
                    name="chevron-forward-outline"></ion-icon>' . app(TranslationService::class)->getDynamicTranslation(Section::class, 'short_description', $section[0]->id, $language_code);
            array_push($right_breadcrumb, $breadcrumb);
        }
        // section filter

        $sortBy = $request->query('sort');
        $bySearch = $request->query('search');

        // by search filter
        if ($bySearch != null) {
            $filter['search'] = $bySearch;
            $breadcrumb = 'Search <ion-icon class="align-text-top icon"
            name="chevron-forward-outline"></ion-icon>' . $bySearch;
            array_push($right_breadcrumb, $breadcrumb);
        }
        // by search filter

        $sort = "";
        $order = "";
        $attribute_values = '';
        $attribute_names = '';
        foreach ($request->query() as $key => $value) {
            if (strpos($key, 'filter-') !== false) {
                if (!empty($attribute_values)) {
                    $attribute_values .= "|" . $request->query($key, true);
                } else {
                    $attribute_values = $request->query($key, true);
                }

                $key = str_replace('filter-', '', $key);
                if (!empty($attribute_names)) {
                    $attribute_names .= "|" . $key;
                } else {
                    $attribute_names = $key;
                }
            }
        }
        $attribute_values = explode('|', $attribute_values ?? '');
        $attribute_names = explode('|', $attribute_names ?? '');
        $filter['attribute_value_ids'] = app(ProductService::class)->getAttributeIdsByValue($attribute_values, $attribute_names);
        // dd($filter['attribute_value_ids']);
        $filter_attribute_value_ids = $filter['attribute_value_ids'];

        // brand filter
        if (isset($request->query()['brand']) && !empty($request->query()['brand'])) {
            $brand_slug = $request->query()['brand'];
            $brand = fetchDetails(Brand::class, ['slug' => $brand_slug, 'status' => '1']);
            $filter['brand'] = $brand[0]->id;
            $breadcrumb = '<a href="' . customUrl('brands') . '"> Brands </a> <ion-icon class="align-text-top icon"
            name="chevron-forward-outline"></ion-icon>' . app(TranslationService::class)->getDynamicTranslation(Brand::class, 'name', $brand[0]->id, $language_code);
            array_push($right_breadcrumb, $breadcrumb);
        }
        // brand filter

        // tags filter
        if (isset($request->query()['tag']) && !empty($request->query()['tag'])) {
            $tags = $request->query()['tag'];
            $filter['tags'] = $tags;
            $breadcrumb = 'Tags<ion-icon class="align-text-top icon"
            name="chevron-forward-outline"></ion-icon>' . $tags;
            array_push($right_breadcrumb, $breadcrumb);
        }
        // tags filter
        // product sort by
        if ($sortBy == "top-rated") {
            $filter['product_type'] = "top_rated_product_including_all_products";
        } elseif ($sortBy == "latest-products") {
            $filter['product_type'] = "new_added_products";
            $sort = 'products.id';
            $order = 'desc';
        } elseif ($sortBy == "oldest-first") {
            $filter['product_type'] = "old_products_first";
            $sort = 'products.id';
            $order = 'asc';
        } elseif ($sortBy == "price-asc") {
            $sort = 'product_variants.price';
            $order = 'asc';
        } elseif ($sortBy == "price-desc") {
            $sort = 'product_variants.price';
            $order = 'desc';
        }
        $sorted_by = "";
        $page = request()->get('page', 1);
        $limit = request()->get('perPage', 20);
        $offset = ($page - 1) * $limit;
        if (isset($request->query()['sort']) && !empty($request->query()['sort'])) {
            $sorted_by = $request->query()['sort'];
        }
        // product sort by
        // In-Stock filter: require stock > 0 and ensure attribute-value-based stock indicators also apply
        if ($request->boolean('in_stock')) {
            // Enforce product/variant stock > 0
            $filter['show_only_stock_product'] = 1;

            // Additionally merge attribute-value ids that denote stock availability (value >= 0)
            $stockAttributeValueIds = Attribute_values::where('value', '>=', 0)
                ->pluck('id')
                ->toArray();

            if (!empty($stockAttributeValueIds)) {
                if (!isset($filter['attribute_value_ids'])) {
                    $filter['attribute_value_ids'] = [];
                }
                $filter['attribute_value_ids'] = array_merge(
                    $filter['attribute_value_ids'],
                    $stockAttributeValueIds
                );
            }
        }
        // min max price filter
        if (isset($request->query()['min_price']) && ($request->query()['min_price'] != null) && isset($request->query()['max_price']) && ($request->query()['max_price'] != null)) {
            // Incoming query is in display currency; convert back to base for filtering
            $filter['min_price'] = $request->query()['min_price'] / $exchange_rate;
            $filter['max_price'] = $request->query()['max_price'] / $exchange_rate;
            // Also support minimum_price/maximum_price for backward compatibility
            $filter['minimum_price'] = $filter['min_price'];
            $filter['maximum_price'] = $filter['max_price'];
        }

        // multiple price ranges filter
        if (!empty($request->query('price_ranges'))) {
            $ranges = explode(',', $request->query('price_ranges'));
            $parsed_ranges = [];
            foreach ($ranges as $range) {
                $parts = explode('-', $range);
                if (count($parts) == 2) {
                    $parsed_ranges[] = [
                        'min' => (float)$parts[0] / $exchange_rate,
                        'max' => (float)$parts[1] / $exchange_rate
                    ];
                }
            }
            $filter['price_ranges'] = $parsed_ranges;
        }
        // min max price filter
        $brands = fetchDetails(Brand::class, ['store_id' => $store_id, 'status' => '1']);
        foreach ($brands as $brand) {
            $brand->is_checked = false;
            if (isset($filter['brand']) && !empty($filter['brand'])) {
                $is_checked = ($brand->id == $filter['brand']) ? true : false;
                $brand->is_checked = $is_checked;
            }
        }
        $product_list = app(ProductService::class)->fetchProduct($this->user_id, $filter, null, $category_id, $limit, $offset, $sort, $order, null, null, null, null, $store_id);
        // dd($product_list);

        if (isset($request->query()['min_price']) && ($request->query()['min_price'] != null) && isset($request->query()['max_price']) && ($request->query()['max_price'] != null)) {
            // Persist the user-entered (display currency) selection
            $selected_min_price = $request->query()['min_price'];
            $selected_max_price = $request->query()['max_price'];
        }
        // Convert base min/max to display currency for the slider
        // Ensure we have valid prices (not 0 or empty)
        $base_max_price = isset($product_list['max_price']) && $product_list['max_price'] > 0
            ? $product_list['max_price']
            : 0;

        $display_min_price = 0;
        $display_max_price = $base_max_price > 0
            ? app(\App\Services\CurrencyService::class)->currentCurrencyPrice($base_max_price, false)
            : 0;

        // $display_min_price = $base_min_price > 0 ? app(\App\Services\CurrencyService::class)->currentCurrencyPrice($base_min_price, false) : 0;
        // $display_max_price = $base_max_price > 0 ? app(\App\Services\CurrencyService::class)->currentCurrencyPrice($base_max_price, false) : 0;

        // Round to 2 decimal places for display but keep precision
        $display_min_price = round($display_min_price, 2);
        $display_max_price = round($display_max_price, 2);

        $min_max_price = [
            'min_price' => $display_min_price,
            'max_price' => $display_max_price,
            'selected_min_price' => $selected_min_price ?? $display_min_price,
            'selected_max_price' => $selected_max_price ?? $display_max_price
        ];

        // product filter by attributes
        $product_attributes = Product::with('productAttributes:id,product_id,attribute_value_ids')
            ->where('store_id', $store_id)
            ->get(['id'])
            ->map(function ($product) {
                return [
                    'id' => $product->id,
                    'attribute_value_ids' => optional($product->productAttributes->first())->attribute_value_ids,
                ];
            })
            ->toArray();
        $attr_val_ids = [];
        foreach ($product_attributes as $product_attr) {
            $att_value_ids = $product_attr['attribute_value_ids'];
            array_push($attr_val_ids, $att_value_ids);
        }
        $attr_val_ids = array_filter(array_merge(...array_map('str_getcsv', $attr_val_ids)));
        $attr_val_ids = array_unique($attr_val_ids);

        $attributeData = Attribute_values::whereIn('attribute_values.id', $attr_val_ids)
            ->join('attributes as a', 'attribute_values.attribute_id', '=', 'a.id')
            ->select('attribute_values.id as attribute_value_id', 'a.name as attribute_name', 'attribute_values.value as attribute_values', 'attribute_values.swatche_type', 'attribute_values.swatche_value')
            ->where('a.status', 1)
            ->get()->toArray();
        $groupedAttributes = [];
        foreach ($attributeData as $item) {
            $attributeName = $item['attribute_name'];
            if (!isset($groupedAttributes[$attributeName])) {
                $groupedAttributes[$attributeName] = [
                    'attribute_name' => $attributeName,
                    'attribute_values' => [],
                    'swatche_type' => [],
                    'swatche_value' => []
                ];
            }
            $groupedAttributes[$attributeName]['attribute_values'][] = $item['attribute_values'];
            $groupedAttributes[$attributeName]['swatche_type'][] = $item['swatche_type'];
            $groupedAttributes[$attributeName]['swatche_value'][] = $item['swatche_value'];
            $is_checked = in_array($item['attribute_value_id'], $filter_attribute_value_ids);
            $groupedAttributes[$attributeName]['is_checked'][] = $is_checked;
        }
        // product filter by attributes

        $bread_crumb['page_main_bread_crumb'] = '<a href="' . customUrl('products') . '">' . labels('front_messages.products', 'Products') . '</a>';

        if (count($right_breadcrumb) >= 1) {
            $bread_crumb['right_breadcrumb'] = $right_breadcrumb;
        }

        // per page
        $perPage = request()->get('perPage', 20);
        if (isset($request->query()['perPage']) && ($request->query()['perPage'] != null)) {
            $perPage = $request->query()['perPage'];
            $perPage = (int)($perPage);
            if ($perPage == 0) {
                $perPage = 20;
            }
        }
        if (!in_array($perPage, [12, 16, 20, 24])) {
            $perPage = 20;
        }
        // per page

        $products = collect($product_list['product']);
        // dd($products);
        $page = request()->get('page', 1);
        // dd($page);

        $total = $product_list['total'] ?? count($product_list['product']);
        if (isset($page)) {
            $paginator = new LengthAwarePaginator(
                $products,
                // $product_list['total'],
                $total,
                $perPage,
                $page,
                ['path' => url()->current()]
            );

        }
        $product_list['product'] = $paginator->items();
        // dd($product_list['product']);
        $product_list['links'] = $paginator->links();

        $relative_products = [];
        if ($total == 0) {
            $relative_product_data = app(ProductService::class)->fetchProduct(
                user_id: $this->user_id,
                limit: 10,
                sort: 'products.rating',
                order: 'DESC',
                store_id: $store_id
            );
            $relative_products = $relative_product_data['product'] ?? [];
        }

        $language_code = app(TranslationService::class)->getLanguageCode();
        $view_mode = $request->query('mode') ?? "";
        return view(
            'livewire.' . config('constants.theme') . '.products.listing',
            [
                'products_listing' => $products,
                // 'total_products' => $product_list['total'],
                'total_products' => $total,
                'min_max_price' => $min_max_price,
                'links' => $product_list['links'],
                'Attributes' => $groupedAttributes,
                'filters' => $filter,
                'bySearch' => $bySearch,
                'sub_categories' => $sub_categories,
                'sorted_by' => $sorted_by,
                'brands' => $brands,
                'filter_categories' => $filterCategories,
                'bread_crumb' => $bread_crumb,
                'view_mode' => $view_mode,
                'perPage' => $perPage,
                'products_type' => "regular",
                'language_code' => $language_code,
                'relative_products' => $relative_products
            ]
        );
    }
}
