type Args<T extends Function> = T extends (...args: infer R) => any ? R : never;
type AnyFunction<T = any> = (...args: T[]) => any;
type Extractable = {
    [key: string]: any;
} | undefined;
type Iteratee<T> = ((value: T) => any) | keyof T;
/**
 * Capitalizes the first character of a string and converts the rest of the string to lowercase.
 *
 * @param s - The string to capitalize.
 * @returns The capitalized string, or an empty string if the input is falsy.
 *
 * @example
 * capitalize('hello'); // returns 'Hello'
 * capitalize(''); // returns ''
 */
declare const capitalize: (s: string) => string;
/**
 * Creates a function that invokes each provided function with the same argument, until
 * one of the functions calls `event.preventDefault()`.
 *
 * @param fns - An array of functions that may or may not be defined.
 * @returns A function that takes an event and invokes each handler with this event.
 *
 * @typeParam T - A function type that takes an event-like argument.
 *
 * @example
 * const handler1 = event => console.log('Handled by first', event.type);
 * const handler2 = event => event.preventDefault();
 * const allHandlers = callAllHandlers(handler1, handler2);
 * allHandlers({ type: 'click' });
 */
declare function callAllHandlers<T extends (event: any) => void>(...fns: (T | undefined)[]): (event: Args<T>[0]) => void;
/**
 * Creates a function that invokes each provided function with the same argument.
 *
 * @param fns - An array of functions that may or may not be defined.
 * @returns A function that takes one argument and invokes all provided functions with it.
 *
 * @typeParam T - A function type that takes any argument.
 *
 * @example
 * const greet = name => console.log(`Hello, ${name}!`);
 * const bye = name => console.log(`Goodbye, ${name}!`);
 * const greetAndBye = callAll(greet, bye);
 * greetAndBye('Alice');
 */
declare function callAll<T extends AnyFunction>(...fns: (T | undefined)[]): (arg: Args<T>[0]) => void;
/**
 * Extracts a property from a list of objects, returning the first found non-falsy value or a default value.
 *
 * @param key - The key of the property to extract.
 * @param defaultValue - The default value to return if no non-falsy property value is found.
 * @param objs - An array of objects to search.
 * @returns The value of the extracted property or the default value.
 *
 * @typeParam K - The type of the key.
 * @typeParam D - The type of the default value.
 *
 * @example
 * extractProperty('name', 'Unknown', { name: 'Alice' }, { name: 'Bob' }); // returns 'Alice'
 * extractProperty('age', 18, { name: 'Alice' }); // returns 18
 */
declare function extractProperty<K extends keyof Extractable, D extends keyof Extractable>(key: K | string, defaultValue: D | string | boolean, ...objs: Extractable[]): string | boolean | D | Extractable[K];
/**
 * Generates a unique identifier using a specified prefix and a random number.
 *
 * @param prefix - The prefix to prepend to the unique identifier.
 * @returns A string that combines the prefix and a random number.
 *
 * @example
 * getUniqueID('btn'); // returns 'btn-123456'
 */
declare function getUniqueID(prefix: string): string;
/**
 * Removes all properties from an object that start with 'on', which are typically event handlers.
 *
 * @param input - The object from which to remove event properties.
 * @returns The same object with event properties removed.
 *
 * @example
 * removeEvents({ onClick: () => {}, onChange: () => {}, value: 10 }); // returns { value: 10 }
 */
declare function removeEvents(input: {
    [key: string]: any;
}): {
    [key: string]: any;
};
/**
 * Converts an object into a JSON string. Returns an empty string if the object
 * is not extractable or if a circular reference is detected during stringification.
 *
 * @param obj - The object to convert into a dependency string.
 *
 * @returns A JSON string representation of the object or an empty string if conversion fails.
 *
 * @example
 * objectToDeps({ key: 'value' }); // returns '{"key":"value"}'
 * objectToDeps(undefined); // returns ""
 */
declare function objectToDeps(obj: Extractable): string;
/**
 * Creates a debounced function that delays invoking `func` until after `waitMilliseconds` have elapsed
 * since the last time the debounced function was invoked. The debounced function has the
 * same `this` context and arguments as the original function.
 *
 * @param func - The function to debounce.
 * @param waitMilliseconds - The number of milliseconds to delay; defaults to 0.
 *
 * @returns A new debounced function.
 *
 * @typeParam F - The type of the function to debounce.
 *
 * @example
 * const save = debounce(() => console.log('Saved!'), 300);
 * save(); // Will log 'Saved!' after 300ms, subsequent calls within 300ms will reset the timer.
 */
declare function debounce<F extends (...args: any[]) => void>(func: F, waitMilliseconds?: number): (this: ThisParameterType<F>, ...args: Parameters<F>) => void;
/**
 * Returns a new array of unique elements from the given array, where the uniqueness is determined by the specified iteratee.
 *
 * @param arr - The array to process.
 * @param iteratee - The iteratee invoked per element to generate the criterion by which uniqueness is computed.
 *                  This can be a function or the string key of the object properties.
 *
 * @returns A new array of elements that are unique based on the iteratee function.
 *
 * @typeParam T - The type of elements in the input array.
 *
 * @example
 * uniqBy([{ id: 1 }, { id: 2 }, { id: 1 }], 'id'); // returns [{ id: 1 }, { id: 2 }]
 */
declare function uniqBy<T>(arr: T[], iteratee: any): T[];
/**
 * Creates an object composed of the own and inherited enumerable property paths of `obj` that are not omitted.
 *
 * @param obj - The source object.
 * @param keys - The property keys to omit.
 *
 * @returns A new object with the keys specified omitted.
 *
 * @typeParam Obj - The type of the object.
 * @typeParam Keys - The type of the keys to omit.
 *
 * @example
 * omit({ a: 1, b: '2', c: 3 }, ['a', 'c']); // returns { b: '2' }
 */
declare const omit: <Obj, Keys extends keyof Obj>(obj: Obj, keys: Keys[]) => Omit<Obj, Keys>;
/**
 * Converts a string to kebab-case.
 *
 * @param s - The string to convert.
 *
 * @returns The kebab-case version of the string.
 *
 * @example
 * kebabCase('fooBar'); // returns 'foo-bar'
 */
declare const kebabCase: (s: string) => string;
/**
 * Creates an object with keys transformed using provided `iteratee` function, which takes each value and the corresponding key.
 *
 * @param obj - The source object.
 * @param iteratee - The function invoked per iteration to transform the keys.
 *
 * @returns A new object with keys transformed by `iteratee`.
 *
 * @example
 * mapKeys({ a: 1, b: 2 }, (value, key) => key + value); // returns { a1: 1, b2: 2 }
 */
declare const mapKeys: (obj: Record<string, any>, iteratee: (value: any, key: string) => any) => Record<string, any>;
/**
 * Retrieves the value at a given path of a provided object safely. If the path does not exist,
 * the function returns a default value, if provided.
 *
 * @param object - The object from which to retrieve the property.
 * @param path - A dot notation string or an array of strings and numbers indicating the path of the property in the object.
 *               Dot notation can also include array indices in bracket notation (e.g., 'a.b[0].c').
 * @param defaultValue - The value to return if the resolved value is `undefined`. This parameter is optional.
 *
 * @returns The value from the object at the specified path, or the default value if the path is not resolved.
 *
 * @example
 * const obj = { a: { b: [{ c: 3 }] } };
 *
 * // Using string path with dot and bracket notation
 * get(obj, 'a.b[0].c'); // returns 3
 *
 * // Using array path
 * get(obj, ['a', 'b', 0, 'c']); // returns 3
 *
 * // Using default value for non-existent path
 * get(obj, 'a.b[1].c', 'not found'); // returns 'not found'
 */
declare const get: (object: Record<string, any>, path: string | (string | number)[], defaultValue?: any) => any;
/**
 * Computes the list of values that are the intersection of all provided arrays,
 * with each element being transformed by the given iteratee, which can be a function or property name.
 *
 * @param args - A rest parameter that collects all arrays to intersect followed by an iteratee.
 * The last element in `args` is the iteratee, which can be either a function or a property name string.
 * The rest are arrays of elements of type T.
 *
 * @returns An array of elements of type T that exist in all arrays after being transformed by the iteratee.
 *
 * @throws {Error} If less than two arguments are provided or if the iteratee is not a function or a valid property string.
 *
 * @typeParam T - The type of elements in the input arrays.
 *
 * @example
 * // Using a function as an iteratee
 * intersectionBy([2.1, 1.2], [2.3, 3.4], Math.floor); // returns [2.1]
 *
 * // Using a property name as an iteratee
 * intersectionBy([{ x: 1 }, { x: 2 }], [{ x: 1 }], 'x'); // returns [{ x: 1 }]
 */
declare const intersectionBy: <T>(...args: [...arrays: T[][], iteratee: Iteratee<T>]) => T[];
/**
 * Checks if the current React version is 19.x.x
 *
 * @returns {boolean} - Returns `true` if the React major version is 19, otherwise returns `false`.
 */
declare const isReact19: () => boolean;
/**
 * Returns an appropriate value for the `inert` attribute based on the React version.
 *
 * In React 19, the attribute `inert` is a boolean. In versions prior to 19, the attribute
 * behaves differently: setting `inert=""` will make it `true`, and `inert=undefined` will make it `false`.
 *
 * @param {boolean} v - The desired boolean state for the `inert` attribute.
 * @returns {boolean | string | undefined} - Depending on the React version:
 * - Returns `boolean` if React version is 19 (the input value `v` directly).
 * - Returns `string` (empty string) if `v` is `true` in older React versions.
 * - Returns `undefined` if `v` is `false` in older React versions.
 *
 * @see {@link https://github.com/facebook/react/issues/17157} for more details on the behavior in older React versions.
 */
declare const getInertValue: (v: boolean) => boolean | string | undefined;

export { callAll, callAllHandlers, capitalize, debounce, extractProperty, get, getInertValue, getUniqueID, intersectionBy, isReact19, kebabCase, mapKeys, objectToDeps, omit, removeEvents, uniqBy };
